/*                                                            */
/*  parser.c -- convert Bourne shell program to PADEL         */
/*                                                            */
/*  sh2pad  by S. Yoshida                                     */
/*                                                            */

#include "config.h"

#include <string.h>
#ifdef HAVE_MALLOC_H
# include <malloc.h>
#endif
#include <ctype.h>
#include "sh2pad.h"
#include "lib.h"

#define MAKE_STRING_WHILE(X)	{\
				    count_paren = 0;\
				    cat_flag_tmp = 0;\
				    key_tmp[0] = '\0';\
				    if(token == LL)\
					cat_flag_tmp = 1;\
				    if(token == Lpar  ||  token == Lbrace)\
					count_paren--;\
				    if(token == Rpar  ||  token == Rbrace)\
					count_paren++;\
				    set_str();\
				    get_token();\
				    while(X){\
					if(token == Ret){\
					    append(" ");\
					    *sp = '\0';\
					    do{\
						get_token();\
					    }while(token == Sp);\
					}else{\
					    if(token == Sp){\
						strcat(sp,word);\
						if(cat_flag_tmp == 2)\
						    cat_flag_tmp = 0;\
					    }else{\
						if(token == Lpar  ||  token == Lbrace)\
						     count_paren--;\
						if(token == Rpar  ||  token == Rbrace)\
						     count_paren++;\
						if(cat_flag_tmp == 1)\
						    cat_flag_tmp = 2;\
						if(cat_flag_tmp == 2)\
						    strcat(key_tmp,word);\
						if(token == LL)\
						    cat_flag_tmp = 1;\
						if(*sp != '\0')\
						    append(sp);\
						*sp = '\0';\
						append_str();\
					    }\
					    get_token();\
					}\
				    }\
				    if(*sp != '\0')\
					append(sp);\
				    *sp = '\0';\
				}

#define END_OF_LINE()		{\
				    tt = token; get_token();\
				    if(strlen(key)==0){\
					while(token == Sp)\
					    get_token();\
					if(tt != Ret  &&  token == Ret)\
					    get_token();\
				    }\
				}

static char *str = NULL;
static int nest_level = 0;
static int count_paren = 0;
static bool elseif_flag = Off;
static token_type tt = Nontoken;
static char key_tmp[BUFSIZ] = {'\0'};
static int cat_flag_tmp = 0;

static void body(void);
static void command(void);
static void function_statement(void);
static void for_statement(void);
static void if_statement(void);
static void case_statement(void);
static void condition(void);

static void set_str(void);
static void append_str(void);
static void append(char *);
static void print_str(void);
static void print_hsp(void);

void parser(void)
/* output */
{
    STR *p,*q;

#ifdef DEBUG
    if(quiet == Off)
	fprintf(stderr,"Translating Bourne shell program into PADEL starts...");
#endif

    puts("{");
    get_token();
    body();
    puts("}");

    if (child != NULL) {	/* when child function declaration is present */
	for (p = child->strr ; p != NULL ; p = q) {
	    q = p->next;
	    fputs(p->str,stdout);
	    free(p->str);
	    free(p);
	}
	free(child);
    }

#ifdef DEBUG
    if(quiet == Off)
	fprintf(stderr,"done\n");
#endif
}

void body(void)
{
    char sp[BUFSIZ] = {'\0'};

    while(token != Eof){
	if (cat_flag > 0) {	/* cat << EOF  */
	    print("{\n");
	    while (cat_flag > 0  &&  token != Eof) {
		MAKE_STRING_WHILE(token != Ret  &&  token != Eof);
		if (strcmp(str,key) == 0) {
		    key[0] = '\0';
		    cat_flag = 0;
		    free(str);
		    str = NULL;
		} else {
		    print("> ");
		    print_str();
		    print("\n");
		}
		get_token();
	    }
	    print("}\n");
	}

	while(token == Sp)
	    get_token();

	if (token == Else  ||  token == Elif  ||  token == Fi  ||  token == Done  ||  token == Esac  ||  token == Wsemicolon  ||  token == Rbrace)
	    return;

	command();
    }
}

void command(void)
{
    char sp[BUFSIZ] = {'\0'};
    char buf[BUFSIZ];

    switch(token){
	case For:
	case While:
	    for_statement();
	    break;
	case If:
	    if_statement();
	    break;
	case Case:
	    case_statement();
	    break;
	case Ret:
	    print("\n");
	    get_token();
	    break;
	case Semicolon:
	    get_token();
	    if(token != Wsemicolon){
		print_hsp();
		print("{    }\n");
		while(token == Sp)
		    get_token();
		if(token == Ret)
		    get_token();
	    }
	    break;
	case Plbrace:
	    print("\n");
	    END_OF_LINE();
	    nest_level++;
	    body();		/* repetition body */
	    nest_level--;
	    END_OF_LINE();
	    break;
	case Function:
	    get_token();
	    while(token == Sp  ||  token == Ret)
		get_token();
	    MAKE_STRING_WHILE(token != Plbrace);
	    function_statement();
	    break;
	default:
	    MAKE_STRING_WHILE(((token != Semicolon  &&  token != Ret  &&  token != Rpar  &&  token != Rbrace  &&  token != Plbrace  &&  token != And2  &&  token != Or2)  ||  count_paren < 0)  &&  token != Eof);
	    /* cat << EOF */
	    if (strlen(key_tmp) > 0) {
		if (key_tmp[0] == '"'  ||  key_tmp[0] == '\'') {
		    key_tmp[strlen(key_tmp) - 1] = '\0';
		    strcpy(key,key_tmp+1);
		} else {
		    strcpy(key,key_tmp);
		}
	    }
	    if (token == Ret)
		cat_flag = strlen(key);

	    print_hsp();
	    if(token == Rpar  ||  token == Rbrace){
		if(global_count_paren > 0){	/* condition in case statement */
		    print("case ");
		    print_str();
		    print(":\n");
		    get_token();
		    while(token == Sp  ||  token == Ret)
			get_token();
		    global_count_paren--;

		    nest_level++;
		    body();		/* repetitino body */
		    nest_level--;

		    if (token == Esac)
			return;
		}else{
		    print_str();
		    print("\n");
		    return;
		}
	    }else if(token == And2  ||  token == Or2){	/* condition evaluation */
		sprintf(buf,"%s )",word);

		get_token();
		while (token == Sp  ||  token == Ret) {
		    get_token();
		}

		print("if ( ");
		print_str();
		print(buf);

		if (token == Plbrace) {
		    print(" {\n");
		    END_OF_LINE();
		    nest_level++;
		    body();		/* repetition body */
		    nest_level--;
		    print_hsp();
		    print("}\n");
		} else {
		    print("\n");
		    nest_level++;
		    command();		/* repetition body */
		    nest_level--;
		    return;
		}
	    }else if(token == Plbrace){		/* function declaration */
		function_statement();
	    }else{
		print_str();
		print("\n");
	    }
	    END_OF_LINE();
    }
}

void function_statement(void)
{
    STR *p,*q;

    if (nest == NULL) {
	if (child == NULL) {
	    nest = (NEST *)malloc(sizeof(NEST));
	    child = nest;
	    nest->parent = nest->child = NULL;
	    nest->strr = nest->strl = NULL;
	} else {
	    nest = child;
	}
    } else {
	if (nest->child == NULL) {
	    nest->child = (NEST *)malloc(sizeof(NEST));
	    nest->child->parent = nest;
	    nest = nest->child;
	    nest->child = NULL;
	    nest->strr = nest->strl = NULL;
	} else {
	    nest = nest->child;
	}
    }
    nest->nest_level = nest_level;

    print("\n");
    print_str();
    print("\n{\n");
    END_OF_LINE();
    nest_level = 1;
    body();
    print("}\n");
    END_OF_LINE();

    if (nest->child != NULL) {	/* when child function declaration is present */
	for (p = nest->child->strr ; p != NULL ; p = q) {
	    q = p->next;
	    print(p->str);
	    free(p->str);
	    free(p);
	}
	free(nest->child);
	nest->child = NULL;
    }
    nest_level = nest->nest_level;
    nest = nest->parent;
}

void for_statement(void)
{
    int i;
    char sp[BUFSIZ] = {'\0'};
    token_type type;

    type = token;

    set_str();
    print_hsp();
    print_str();

    get_token();
    while (token == Sp) {
	strcat(sp,word);
	get_token();
    }
    print(sp);

    print("(");
    MAKE_STRING_WHILE(token != Semicolon  &&  token != Ret  &&  token != Do  &&  token != Eof);
    i = strlen(str) - 1;
    while(isspace(str[i])  &&  i >= 0)
	str[i--] = '\0';
    print_str();
    print(")");

    sp[0] = '\0';
    while (token != Do  &&  token != Eof) {
	strcat(sp,(*word == '\t' ? "\t" : " "));
	get_token();
    }
    print(sp);
    print("{\n");

    while (token != Ret  &&  token != Semicolon  &&  token != Eof)
	get_token();
    END_OF_LINE();

    nest_level++;
    body();				/* repetition body */
    nest_level--;

    print_hsp();
    print("}\n");

    while (token != Ret  &&  token != Semicolon  &&  token != Eof)
	get_token();
    END_OF_LINE();
}

void if_statement(void)
{
    char sp[BUFSIZ] = {'\0'};
    char buf[BUFSIZ];

    set_str();

    if(elseif_flag == On)	/* output already printed for elif */
	elseif_flag = Off;
    else {
	print_hsp();
	print_str();
    }

    get_token();
    while (token == Sp) {
	strcat(sp,word);
	get_token();
    }
    print(sp);

    if (token == Lbracket) {
	print("(");
	condition();
	print_str();
    } else {
	print("(");
	MAKE_STRING_WHILE(token != Semicolon  &&  token != Ret  &&  token != Eof);
	print_str();
    }
    print(")");
    no_print_comment_flag = On;
    while (token != Ret  &&  token != Semicolon  &&  token != Comment  &&  token != Eof)
	get_token();
    no_print_comment_flag = Off;

    if (token == Ret  ||  token == Comment) {
	putchar('\n');
	if (token == Comment)
	    puts(word);

	END_OF_LINE();
	while (token == Sp) {
	    get_token();
	}

	print_hsp();
    } else {
	END_OF_LINE();

	sp[0] = '\0';
	while (token == Sp) {
	    strcat(sp,word);
	    get_token();
	}

	print(sp);
    }
    print("{\n");
    END_OF_LINE();

    nest_level++;
    body();
    nest_level--;

    print_hsp();
    if (token == Fi) {
	print("}\n");
	while (token != Ret  &&  token != Semicolon  &&  token != Eof)
	    get_token();
	END_OF_LINE();
    } else if (token == Elif) {
	sprintf(buf,"}%selse if",sp);
	print(buf);
	elseif_flag = On;
	if_statement();
    } else {
	sprintf(buf,"}%selse%s{\n",sp,sp);
	print(buf);
	MAKE_STRING_WHILE(token != Semicolon  &&  token != Ret  &&  token != Eof);
	END_OF_LINE();

	nest_level++;
	body();
	nest_level--;

	print_hsp();
	print("}\n");

	while (token != Ret  &&  token != Semicolon  &&  token != Eof)
	    get_token();
	END_OF_LINE();
    }
}

void case_statement(void)
{
    int i;
    char sp[BUFSIZ] = {'\0'};

    print_hsp();
    print("switch");

    get_token();
    while (token == Sp) {
	strcat(sp,word);
	get_token();
    }
    print(sp);

    print("(");
    MAKE_STRING_WHILE(token != In  &&  token != Eof);
    i = strlen(str) - 1;
    while(isspace(str[i])  &&  i >= 0)
	str[i--] = '\0';
    print_str();
    print(")");

    sp[0] = '\0';
    get_token();
    while (token == Sp  ||  token == Ret) {
	strcat(sp,(*word == '\t' ? "\t" : " "));
	get_token();
    }
    print(sp);
    print("{\n");

    nest_level++;
    body();				/* repetitino body */
    nest_level--;

    print_hsp();
    print("}\n");

    while (token != Ret  &&  token != Semicolon  &&  token != Eof)
	get_token();
    END_OF_LINE();
}

void condition(void)
{
    char sp[BUFSIZ] = {'\0'};

    append_str();
    get_token();

    while(token != Rbracket  &&  token != Eof){
	if(token == Lbracket)
	    condition();
	else{
	    if(token == Sp)
		strcat(sp,word);
	    else{
		if(*sp != '\0')
		    append(sp);
		*sp = '\0';
		append_str();
	    }

	    get_token();
	}
    }

    if(*sp != '\0')
	append(sp);
    append_str();
    get_token();
}

void set_str(void)
{
    int l;

    l = strlen(word);
    str = (char *)malloc((l+2) * sizeof(char));
    strcpy(str,word);
}

void append_str(void)
{
    int l;
    char *s;

    l = strlen(word);
    s = (char *)malloc((l+2) * sizeof(char));
    strcpy(s,word);
    str = joint(2,str,s);
}

void append(char *p)
{
    int l;
    char *s;

    l = strlen(p);
    s = (char *)malloc((l+2) * sizeof(char));
    strcpy(s,p);
    str = joint(2,str,s);
}

void print_str(void)
{
    if(str != NULL){
	print(str);
	free(str);
	str = NULL;
    }
}

void print_hsp(void)
{
    int i;

    for(i = 0 ; i < nest_level * tab ; i++)
	print(" ");
}

void print(const char *s)
{
    if (nest == NULL) {
	fputs(s,stdout);
    } else {
	if (nest->strl == NULL) {
	    nest->strr = (STR *)malloc(sizeof(STR));
	    nest->strl = nest->strr;
	} else {
	    nest->strl->next = (STR *)malloc(sizeof(STR));
	    nest->strl = nest->strl->next;
	}
	nest->strl->next = NULL;
	nest->strl->str = (char *)malloc(strlen(s) + 1);
	strcpy(nest->strl->str,s);
    }
}
