/*                               */
/*  ps.c -- output in PostScript */
/*                               */
/*  pad2ps  by S. Yoshida        */
/*                               */

#include "config.h"

#include "pad2ps.h"

#define X(x)	((x) + x_base + x0)
#define Y(y)	(- (y) + y_base)

static int x_base;
static int y_base;
static int x0 = 0;
static int line_w = 0;
static int page_height;
static BOX *back = NULL;
static bool draw_flag = Off;
static bool right_more = Off;
static bool left_more = Off;
static BOX *pagetop_pad;
static char *prev_f = NULL;
static int prev_fw = EOF, prev_fs = EOF;

static void init_ps(void);
static void draw_pad(void);
static void draw_box(void);
static void draw_label(void);
static void draw_if(void);
static void draw_line(int);
static void write(void);
static void draw_init(void);
static void draw_end(void);
static void begin_ps(void);
static void end_ps(void);
static void newpage(void);
static void begin_page(void);

void ps(void)
/* create PostScript */
{
    char tmp[BUFSIZ];

#ifdef DEBUG
    if(quiet == Off){
	if(eps_flag == On)
	    fprintf(stderr,"Making Encapsulated PostScript starts...\n");
	else
	    fprintf(stderr,"Making PostScript starts...\n");
    }
#endif

    if(eps_flag == On){
	sprintf(tmp,"%s.%d.eps",outfile,page + 1);
	fp = fopen(tmp,"w");
    }else{
	if(outfile[0] != '\0')
	    fp = fopen(outfile,"w");
    }

    init_ps();

    begin_ps();
    draw_pad();
    end_ps();

    if(eps_flag == On)
	fclose(fp);
    else{
	if(outfile[0] != '\0')
	    fclose(fp);
    }

    if(quiet == Off  &&  print_warning == Off)
        fputc('\n',stderr);
}

void init_ps(void)
/* initialize */
{
    if(proper_hrate < hrate  &&  poster == Off){	/* adjust sideway magnification */
        char tmp[BUFSIZ];
        hrate = proper_hrate;
	sprintf(tmp,"horizontal overflow -- compressed with hrate=%f.",proper_hrate);
	warning(quiet == Off && print_warning == On ? tmp : NULL);
    }

    if(centering == On)
        x_base = (int)((double)(X_CENTER / hrate));
    else
        x_base = page_left;
    page_left = 0;
    page_right = (int)((double)(X_MAX / hrate));

    page_top = (int)((double)(page_top / vrate));
    page_bottom = (int)((double)(page_bottom / vrate));
    page_height = page_top - page_bottom;
    y_base = page_top;
}

void draw_pad(void)
{
    lp = first;

    begin_page();

    while(lp != NULL  ||  back != NULL  ||
	  (poster == On  &&  (left_more == On  ||  right_more == On)))
    {
	if(lp == NULL  ||  (lp != NULL  &&  Y(lp->y) < page_bottom))
	    /* carriage return if the PAD will be set completely on the next page */
	    newpage();

	if(Y(lp->y + lp->h) <= page_top)	/* skip the PAD if it is out of range */
	{
	    if(Y(lp->y + lp->h) < page_bottom  &&  back == NULL)
		/* when the PAD is split over pages */
	        back = lp;

	    switch(lp->type){
		case Box:
		case Label:
		case If:
		case Line:
		case Padframe:
		    if(X(lp->x) < 0  &&  x0 >= 0)
		        left_more = On;
		    if(X(lp->x + lp->w) > page_right)
		        right_more = On;

		    switch(lp->type){
		        case Box:
		            draw_box();
			    break;
		        case Label:
		            draw_label();
			    break;
			case If:
			    draw_if();
			    break;
			case Line:
			    draw_line(linewidth);
			    break;
			case Padframe:
			    draw_line(framelinewidth);
			    break;
		    }
	    }

	    if(lp->str != NULL){
	        if(draw_flag == On)
		    draw_end();
	        write();
	    }
	}

	lp = lp->next;
    }
}

void draw_box(void)
{
    if(line_w != padlinewidth){		/* change thickness of line */
        if(draw_flag == On)
	    draw_end();
	fprintf(fp,"%d setlinewidth\n",padlinewidth);
	line_w = padlinewidth;
    }

    if(draw_flag == Off)
        draw_init();

    fprintf(fp,"  %d %d %d %d box\n",X(lp->x),Y(lp->y),lp->w,lp->h);

    if(lp->flag1 == On)
        fprintf(fp,"  %d %d %d %d line\n",X(lp->x + sideline_space),Y(lp->y),0,lp->h);
    if(lp->flag2 == On)
        fprintf(fp,"  %d %d %d %d line\n",X(lp->x + lp->w - sideline_space),Y(lp->y),0,lp->h);
}

void draw_label(void)
{
    if(line_w != padlinewidth){		/* change thickness of line */
        if(draw_flag == On)
	    draw_end();
	fprintf(fp,"%d setlinewidth\n",padlinewidth);
	line_w = padlinewidth;
    }

    if(draw_flag == Off)
        draw_init();

    fprintf(fp,"  %d %d %d %d circle\n",X(lp->x),Y(lp->y),lp->w,lp->h);
}

void draw_if(void)
{
    if(line_w != padlinewidth){		/* change thickness of line */
        if(draw_flag == On)
	    draw_end();
	fprintf(fp,"%d setlinewidth\n",padlinewidth);
	line_w = padlinewidth;
    }

    if(draw_flag == Off)
        draw_init();

    if(lp->flag1 == On){
        if(lp->flag2 == On)		/* perfect if box */
	    fprintf(fp,"  %d %d %d %d ifbox\n",X(lp->x),Y(lp->y),lp->w,lp->h);
	else				/* if box without under-line */
	    fprintf(fp,"  %d %d %d %d ifline\n",X(lp->x),Y(lp->y + lp->h),lp->w,-(lp->h));
    }else{
        if(lp->flag2 == On)		/* if box without upper-line */
	    fprintf(fp,"  %d %d %d %d ifline\n",X(lp->x),Y(lp->y),lp->w,lp->h);
	else{				/* if box without upper- and lower-line */
	    fprintf(fp,"  %d %d %d %d line\n",X(lp->x),Y(lp->y),0,lp->h);
	    fprintf(fp,"  %d %d %d %d line\n",X(lp->x + lp->w - if_rightspace),
		    Y(lp->y + lp->h / 2),if_rightspace,-(lp->h / 2));
	    fprintf(fp,"  %d %d %d %d line\n",X(lp->x + lp->w - if_rightspace),
		    Y(lp->y + lp->h / 2),if_rightspace,lp->h / 2);
	}
    }
}

void draw_line(int w)
{
    if(line_w != w){		/* change thickness of line */
        if(draw_flag == On)
	    draw_end();
	fprintf(fp,"%d setlinewidth\n",w);
	line_w = w;
    }

    if(draw_flag == Off)
        draw_init();

    fprintf(fp,"  %d %d %d %d line\n",X(lp->x),Y(lp->y),lp->w,lp->h);
}

void write(void)
/* print string */
{
    STRING *s;
    char *ptr,*f;
    int fw,fs,baseup;

    for(s = lp->str ; s != NULL ; s = s->next)
    {
        if(s->str != NULL){
#ifdef JAPANESE
	    if(s->jflag == On)
	        f = jfont;
	    else
#endif
	        f = font;

	    switch(lp->type){
		case Title:
		    change_font(title_rate);
#ifdef JAPANESE
		    if(s->jflag == On)
		        f = title_jfont;
		    else
#endif
		        f = title_font;
		    break;
		case Padtitle:
		    change_font(padtitle_rate);
		    break;
		case Figure:
		    change_font(figure_rate);
		    break;
	    }

	    fs = fontsize;
	    fw = (int)((double)(fontwidth * 1.67));
	    baseup = CHAR_BASEUP;
#ifdef JAPANESE
	    if(s->jflag == On){
	        fw = jfontwidth;
		baseup = JCHAR_BASEUP;
	    }
#endif

	    if(f != prev_f  ||  fw != prev_fw  ||  fs != prev_fs){
	        fprintf(fp,"/%s findfont [%d 0 0 %d 0 0] makefont setfont\n",f,fw,fs);
		prev_f = f;
		prev_fw = fw;
		prev_fs = fs;
	    }

	    if(lp->type == Title  ||  lp->type == Padtitle  ||  lp->type == Figure)
	        restore_font();

	    putc('(',fp);
	    for(ptr = s->str ; *ptr != '\0' ; ptr++){
	        switch(*ptr){
		    case '(':
		    case ')':
		    case '\\':
			putc('\\',fp);
			break;
		}
		putc(*ptr,fp);
	    }
	    fprintf(fp,") %d %d string\n",X(s->x),Y(s->y + fs - baseup));
	}
    }
}

void draw_init(void)
/* initialize for drawing PAD */
{
    fputs("newpath\n",fp);
    draw_flag = On;
}

void draw_end(void)
/* finish drawing PAD */
{
    fputs("stroke\n",fp);
    draw_flag = Off;
}

void begin_ps(void)
/* print the beginning portion of PostScript */
{
    if(eps_flag == Off){
	fputs("%!PS-Adobe-1.0\n",fp);
	fputs("% pad2ps PostScript output\n",fp);
	fprintf(fp,"%%   %s\n",AUTHOR);
	fputs("%\n",fp);
    }else{
	fputs("%!PS-Adobe-1.0 EPSF-1.0\n",fp);
	fputs("% pad2ps Encapsulated PostScript output\n",fp);
	fprintf(fp,"%%   %s\n",AUTHOR);
	fputs("%\n",fp);
	fputs("%%BoundingBox: 0 49 540 637\n",fp);
    }
    fputs("%\n",fp);
    fputs("clippath pathbbox\n",fp);
    fputs("/page-height exch def\n",fp);
    fputs("/page-width exch def\n",fp);
    fputs("pop pop\n",fp);
    fputs("%\n",fp);
    fputs("/string {\n",fp);
    fputs("   moveto show\n",fp);
    fputs("} def\n",fp);
    fputs("%\n",fp);
    fputs("% PAD line -- usage: x0 y0 w h line\n",fp);
    fputs("/line {\n",fp);
    fputs("   neg 4 -2 roll moveto rlineto\n",fp);
    fputs("} def\n",fp);
    fputs("%\n",fp);
    fputs("% PAD box -- usage: x0 y0 w h box\n",fp);
    fputs("/box {\n",fp);
    fputs("   4 -2 roll moveto\n",fp);
    fputs("   dup neg 0 exch rlineto\n",fp);
    fputs("   exch dup 0 rlineto\n",fp);
    fputs("   exch dup 0 exch rlineto\n",fp);
    fputs("   pop pop\n",fp);
    fputs("   closepath\n",fp);
    fputs("} def\n",fp);
    fputs("%\n",fp);
    fputs("% PAD circle -- usage: x0 y0 w h circle\n",fp);
    fputs("/circle {\n",fp);
    fputs("   dup 3 -1 roll dup 3 -1 roll 6 -1 roll dup 7 -1 roll dup\n",fp);
    fputs("   8 -1 roll 2 idiv sub 3 -1 roll 7 -1 roll add\n",fp);
    fputs("   exch moveto\n",fp);
    fputs("   4 -2 roll dup 3 -1 roll dup 3 1 roll\n",fp);
    fputs("   div dup dup 7 2 roll 1 exch scale\n",fp);
    fputs("   dup 5 1 roll\n",fp);
    fputs("   3 1 roll 2 idiv sub 5 -1 roll div\n",fp);
    fputs("   3 1 roll 2 idiv add\n",fp);
    fputs("   exch 3 -1 roll 2 idiv 0 360 arc\n",fp);
    fputs("   1 exch div 1 exch scale\n",fp);
    fputs("} def\n",fp);
    fputs("%\n",fp);
    fputs("% PAD ifline -- usage: x0 y0 w h ifline\n",fp);
    fputs("/ifline {\n",fp);
    fputs("   4 -2 roll moveto\n",fp);
    fputs("   dup neg 0 exch rlineto\n",fp);
    fputs("   exch dup 0 rlineto\n",fp);
    fprintf(fp,"   exch dup 2 idiv dup -%d exch rlineto\n",if_rightspace);
    fprintf(fp,"   sub %d exch rlineto\n",if_rightspace);
    fputs("   pop\n",fp);
    fputs("} def\n",fp);
    fputs("%\n",fp);
    fputs("% PAD ifbox -- usage: x0 y0 w h ifbox\n",fp);
    fputs("/ifbox {\n",fp);
    fputs("   ifline closepath\n",fp);
    fputs("} def\n",fp);
    fputs("%\n",fp);
    fputs("/clippingpath {\n",fp);
    fputs("   newpath\n",fp);
    fprintf(fp,"   %d %d moveto\n",0,page_top);
    fprintf(fp,"   %d %d lineto\n",0,page_bottom);
    fprintf(fp,"   %d %d lineto\n",(int)((double)(X_MAX / hrate)),page_bottom);
    fprintf(fp,"   %d %d lineto\n",(int)((double)(X_MAX / hrate)),page_top);
    fputs("   closepath\n",fp);
    fputs("} def\n",fp);
    fputs("%\n",fp);
    fputs("/pageframe {\n",fp);
    fprintf(fp,"   %d setlinewidth\n",framelinewidth);
    fputs("   newpath\n",fp);
    fprintf(fp,"   %d %d moveto\n",0,page_top);
    fprintf(fp,"   %d %d lineto\n",0,page_bottom);
    fprintf(fp,"   %d %d lineto\n",(int)((double)(X_MAX / hrate)),page_bottom);
    fprintf(fp,"   %d %d lineto\n",(int)((double)(X_MAX / hrate)),page_top);
    fputs("   closepath stroke\n",fp);
    fputs("} def\n",fp);
    fputs("%\n",fp);
    fprintf(fp,"/%s findfont %d scalefont setfont\n",font,fontsize);
}

void end_ps(void)
/* print the last portion of PostScript */
{
    if(draw_flag == On)
        draw_end();

    fputs("showpage\n",fp);
    fputs("clear\n",fp);
    fputs("%%PageTrailer\n",fp);
    fprintf(fp,"%%%%Pages: %d\n",page);
    fputs("% End of pad2ps PostScript output\n",fp);
}

void newpage(void)
/* new page */
{
    char tmp[BUFSIZ];

    if(left_more == Off  &&  right_more == Off)
        y_base += page_height;

    if(draw_flag == On)
	draw_end();

    if(eps_flag == On){
	end_ps();
	fclose(fp);

	sprintf(tmp,"%s.%d.eps",outfile,page + 1);
	fp = fopen(tmp,"w");
	begin_ps();
    }else{
	fputs("showpage\n",fp);
	fputs("clear\n",fp);
	fputs("%\n",fp);
    }

    if(back != NULL)
        lp = back;
    back = NULL;

    begin_page();

    prev_f = NULL;
    prev_fw = EOF;
    prev_fs = EOF;
}

void begin_page(void)
/* begin new page */
{
    line_w = 0;
    page++;

    if(left_more == On){		/* shift to left */
        x0 += page_right;
	left_more = Off;
	lp = pagetop_pad;
    }else if(right_more == On){		/* shift to right */
        if(x0 > 0)		/* if already shifted to left, return to the original position */
	    x0 = - page_right;
	else
	    x0 -= page_right;
	right_more = Off;
	lp = pagetop_pad;
    }else{
        x0 = 0;
	pagetop_pad = lp;
    }

    fputs("%\n",fp);
    fputs("%%PageTrailer\n",fp);
    fprintf(fp,"%%%%Page: %d %d\n",page,page);
    if(quiet == Off)
        fprintf(stderr,"[%d]",page);
    fputs("%\n",fp);
    fputs("%\n",fp);
    fputs("% newpage\n",fp);
    fputs("%\n",fp);
    fputs("page-width 538.583 sub 2 div\n",fp);
    fputs("page-height 538.583 sub 2 div\n",fp);
    fputs("translate\n",fp);
    fputs("%\n",fp);
    fprintf(fp,"%g %g scale\n",SCALE * hrate,SCALE * vrate);
    fputs("%\n",fp);
    if(page_print == On){
        fprintf(fp,"/Times-Roman findfont [%d 0 0 %d 0 0] makefont setfont\n",
	       (int)((double)(FONTSIZE / hrate)),(int)((double)(FONTSIZE / vrate)));
        fprintf(fp,"(%d) %d %d string\n",page,
	        (int)((double)(X_CENTER - FONTSIZE) / hrate),
	        (int)((double)(PAGENUMBER_Y + FONTSIZE) / vrate));
	fputs("%\n",fp);
	prev_f = NULL;
	prev_fw = EOF;
	prev_fs = EOF;
    }
    fputs("clippingpath clip\n",fp);
    fputs("%\n",fp);
    if(pageframe == On)
        fputs("pageframe\n",fp);
}
