%{
/*                                        */
/*  parser.y -- parser (syntax analysis)  */
/*                                        */
/*  pad2ps  by S. Yoshida                 */
/*                                        */

#include "config.h"

#include <stdarg.h>
#include <string.h>
#ifdef HAVE_MALLOC_H
# include <malloc.h>
#endif
#include <ctype.h>
#include "pad2ps.h"
#include "lib.h"

#ifdef YACC_NO_VOID_PTR
#define YYSTYPE int
#else
#define YYSTYPE void *
#endif

#ifdef DEBUG
#undef YYDEBUG
#define YYDEBUG	1
#endif

static STACK *sw = NULL;
static PAD *tmp_pad = NULL;

static PAD *make_pad(TYPE,char *,PAD *,PAD *);
static void link_pad(PAD *);
static void dup_pad(PAD *);

static char *get_termstr(char *);
static char *get_until(char *,char,int);
static char *get_headsp(char *);
static char *rm_headsp(char *);
static char *rm_tailsp(char *);

static char *complete_label(char *);
static void push_sw(void *);
static void *pop_sw(void);
%}

%token FOR
%token WHILE
%token DO
%token IF
%token ELSE
%token SWITCH
%token CASE
%token DEFAULT_LABEL
%token DEFAULT
%token LABEL
%token REFER
%token SHORT_LABEL
%token TRY
%token CATCH
%token LB_END
%token RB_END
%token RP_END
%token RPLB_END
%token COLON_END
%token RB_ELSE
%token RB_WHILE
%token RB_CATCH
%token LP
%token RP
%token LB
%token RB
%token SP
%token RET
%token CHAR
%token MESSAGE
%token MESSAGE_END
%token COMMENT
%token NP
%token VSP

%%

program
		: 
		| program RET
		{ link_pad(make_pad(Empty,NULL,NULL,NULL)); }
		| program control_code
		{ link_pad($2); }
		| program body
		{ link_pad($2); }
		;
control_code
		: NP
		{ $$ = make_pad(Newpage,NULL,NULL,NULL); }
		| VSP
		{ tmp_pad = make_pad(Vsp,NULL,NULL,NULL);
		  tmp_pad->value = (int)$1; $$ = tmp_pad; }
		;
body
		: LB_END commands RB_END
		{ current = pop(); $$ = make_pad(Padtitle,NULL,$2,NULL); }
		| string end LB_END commands RB_END
		{ current = pop();
		  $$ = make_pad(Padtitle,joint(2,$1,$2),$4,NULL); }
		| one_symbol_line LB_END commands RB_END
		{ current = pop();
		  $$ = make_pad(Padtitle,$1,$3,NULL); }
		;
commands
		:
		{ push(current); $$ = NULL; }
		| real_commands
		{ $$ = $1; }
		;
real_commands
		: command
		{ push(current); current = $1; $$ = $1; }
		| real_commands command
		{ link_pad($2); $$ = $1; }
		;
oneline
		: oneline_command
		{ $$ = $1; }
		;
oneline_command
		: for_statement			{ $$ = $1; }
		| while_statement		{ $$ = $1; }
		| do_statement			{ $$ = $1; }
		| ifelse_statement		{ $$ = $1; }
		| trycatch_statement		{ $$ = $1; }
		| switch_statement		{ $$ = $1; }
		| label_statement		{ $$ = $1; }
		| refer_statement		{ $$ = $1; }
		| oneline_block			{ $$ = $1; }
		| normal_line			{ $$ = $1; }
		;
command
		: oneline_command		{ $$ = $1; }
		| block				{ $$ = $1; }
		| RET
		{ $$ = make_pad(Empty,NULL,NULL,NULL); }
		| message
		{ $$ = make_pad(Message,$1,NULL,NULL); }
		| control_code			{ $$ = $1; }
		;
for_statement
		: FOR sp LP string RP_END LB_END commands RB_END
		{ current = pop(); $$ = make_pad(For,$4,$7,NULL); }
		| FOR sp LP string RPLB_END commands RB_END
		{ current = pop(); $$ = make_pad(For,$4,$6,NULL); }
		| FOR sp LP string RP_END oneline
		{ $$ = make_pad(For,$4,$6,NULL); }
		;
while_statement
		: WHILE sp LP string RP_END LB_END commands RB_END
		{ current = pop(); $$ = make_pad(While,$4,$7,NULL); }
		| WHILE sp LP string RPLB_END commands RB_END
		{ current = pop(); $$ = make_pad(While,$4,$6,NULL); }
		| WHILE sp LP string RP_END oneline
		{ $$ = make_pad(While,$4,$6,NULL); }
		;
do_statement
		: DO LB_END commands RB_WHILE sp WHILE sp LP string RP_END
		{ current = pop(); $$ = make_pad(Do,$9,$3,NULL); }
		| DO LB_END commands RB_END WHILE sp LP string RP_END
		{ current = pop(); $$ = make_pad(Do,$8,$3,NULL); }
		| DO RET LB_END commands RB_WHILE sp WHILE sp LP string RP_END
		{ current = pop(); $$ = make_pad(Do,$10,$4,NULL); }
		| DO RET LB_END commands RB_END WHILE sp LP string RP_END
		{ current = pop(); $$ = make_pad(Do,$9,$4,NULL); }
		| DO RET oneline WHILE sp LP string RP_END
		{ $$ = make_pad(Do,$7,$3,NULL); }
		;
ifelse_statement
		: if_statement
		{ $$ = make_pad(If,NULL,$1,NULL); }
		;
if_statement
		: if_head commands RB_END
		{ current = pop(); $$ = make_pad(Cond,$1,$2,NULL); }
		| if_head commands RB_END else_statement
		{ current = make_pad(Cond,$1,$2,NULL); push(current);
		  link_pad($4); tmp_pad = pop(); current = pop(); $$ = tmp_pad; }
		| if_head commands RB_ELSE sp else_statement
		{ current = make_pad(Cond,$1,$2,NULL); push(current);
		  link_pad($5); tmp_pad = pop(); current = pop(); $$ = tmp_pad; }
		| IF sp LP string RP_END oneline
		{ $$ = make_pad(Cond,$4,$6,NULL); }
		| IF sp LP string RP_END oneline else_statement
		{ push(current); current = make_pad(Cond,$4,$6,NULL);
		  push(current); link_pad($7); tmp_pad = pop(); current = pop();
		  $$ = tmp_pad; }
		;
if_head
		: IF sp LP string RP_END LB_END
		{ $$ = $4; }
		| IF sp LP string RPLB_END
		{ $$ = $4; }
		;
else_statement
		: ELSE SP if_statement
		{ $$ = $3; }
		| ELSE RET LB_END commands RB_END
		{ current = pop(); $$ = make_pad(Cond,NULL,$4,NULL); }
		| ELSE LB_END commands RB_END
		{ current = pop(); $$ = make_pad(Cond,NULL,$3,NULL); }
		| ELSE RET oneline
		{ $$ = make_pad(Cond,NULL,$3,NULL); }
		;
trycatch_statement
		: try_statement catch_statement
		{ $$ = make_pad(If,NULL,$2,NULL); }
		;
try_statement
		: TRY LB_END commands
		{ tmp_pad = current; current = pop();
		  link_pad($3); current = tmp_pad; $$ = NULL; }
		| TRY RET LB_END commands
		{ tmp_pad = current; current = pop();
		  link_pad($4); current = tmp_pad; $$ = NULL; }
		| TRY RET oneline
		{ link_pad($3); $$ = NULL; }
		;
catch_statement
		: RB_END CATCH sp LP string RP_END LB_END commands RB_END
		{ current = pop(); $$ = make_pad(Cond,$5,$8,NULL); }
		| RB_CATCH sp CATCH sp LP string RP_END LB_END commands RB_END
		{ current = pop(); $$ = make_pad(Cond,$6,$9,NULL); }
		| CATCH sp LP string RP_END LB_END commands RB_END
		{ current = pop(); $$ = make_pad(Cond,$4,$7,NULL); }
		| RB_END CATCH sp LP string RPLB_END commands RB_END
		{ current = pop(); $$ = make_pad(Cond,$5,$7,NULL); }
		| RB_CATCH sp CATCH sp LP string RPLB_END commands RB_END
		{ current = pop(); $$ = make_pad(Cond,$6,$8,NULL); }
		| CATCH sp LP string RPLB_END commands RB_END
		{ current = pop(); $$ = make_pad(Cond,$4,$6,NULL); }
		| RB_END CATCH sp LP string RP_END oneline
		{ $$ = make_pad(Cond,$5,$7,NULL); }
		| RB_CATCH sp CATCH sp LP string RP_END oneline
		{ $$ = make_pad(Cond,$6,$8,NULL); }
		| CATCH sp LP string RP_END oneline
		{ $$ = make_pad(Cond,$4,$6,NULL); }
		;
switch_statement
		: switch_head switch_blocks RB_END
		{ pop_sw(); current = pop();
		  $$ = make_pad(Switch,NULL,$2,NULL); }
		;
switch_head
		: SWITCH sp LP string RP_END LB_END
		{ push_sw($4); }
		| SWITCH sp LP string RPLB_END
		{ push_sw($4); }
		| SWITCH LB_END
		{ push_sw(NULL); }
		| SWITCH RET LB_END
		{ push_sw(NULL); }
		;
switch_blocks
		:
		{ push(current); $$ = NULL; }
		| switch_label
		{ push(current); $$ = make_pad(Cond,$1,NULL,NULL); }
		| switch_label switch_labels
		{ push(current); $$ = make_pad(Cond,$1,NULL,$2); }
		| real_switch_blocks
		{ $$ = $1; }
		| real_switch_blocks switch_label
		{ link_pad(make_pad(Cond,$2,NULL,NULL)); $$ = $1; }
		| real_switch_blocks switch_label switch_labels
		{ current = pop(); link_pad(make_pad(Cond,$2,NULL,$3)); $$ = $1; }
		;
real_switch_blocks
		: switch_block
		{ push(current); current = $1; $$ = $1; }
		| real_switch_blocks switch_block
		{ link_pad($2); $$ = $1; }
		;
switch_block
		: switch_label real_commands
		{ current = pop(); $$ = make_pad(Cond,$1,$2,NULL); }
		| switch_label switch_labels real_commands
		{ pop(); current = pop(); $$ = make_pad(Cond,$1,$3,$2); }
		;
switch_labels
		: switch_label
		{ push(current); current = make_pad(Cond,$1,NULL,NULL);
		  $$ = current; }
		| switch_labels switch_label
		{ dup_pad(make_pad(Cond,$2,NULL,NULL)); $$ = $1; }
		;
switch_label
		: case_statement
		{ $$ = $1; }
		| default_statement
		{ $$ = $1; }
		;
case_statement
		: CASE string COLON_END
		{ $$ = complete_label($2); }
		;
default_statement
		: DEFAULT_LABEL
		{ free($1); $$ = NULL; }
		;
label_statement
		: LABEL string end
		{ $$ = make_pad(Label,joint(2,$2,$3),NULL,NULL); }
		| LABEL string end comment
		{ $$ = make_pad(Label,joint(2,$2,$3),make_pad(Comment,$4,NULL,NULL),NULL); }
		| LABEL sp one_symbol_line
		{ $$ = make_pad(Label,$3,NULL,NULL); }
		| LABEL sp one_symbol_line comment
		{ $$ = make_pad(Label,$3,make_pad(Comment,$4,NULL,NULL),NULL); }
		| SHORT_LABEL
		{ $$ = make_pad(Label,rm_headsp(rm_tailsp(get_until($1,':',0))),NULL,NULL); }
		| SHORT_LABEL comment
		{ $$ = make_pad(Label,rm_headsp(rm_tailsp(get_until($1,':',0))),make_pad(Comment,$2,NULL,NULL),NULL); }
		;
refer_statement
		: REFER sp LP string RP_END LB_END block_strings RB_END
		{ current = pop(); $$ = make_pad(Refer,$4,$7,NULL); }
		| REFER sp LP string RP_END LB_END block_strings RB_END comment
		{ current = pop(); $$ = make_pad(Refer,$4,$7,make_pad(Comment,$9,NULL,NULL)); }
		| REFER sp LP string RPLB_END block_strings RB_END
		{ current = pop(); $$ = make_pad(Refer,$4,$6,NULL); }
		| REFER sp LP string RPLB_END block_strings RB_END comment
		{ current = pop(); $$ = make_pad(Refer,$4,$6,make_pad(Comment,$8,NULL,NULL)); }
		| REFER sp LP string RP_END block_string
		{ $$ = make_pad(Refer,$4,$6,NULL); }
		| REFER sp LP string RP_END block_string comment
		{ $$ = make_pad(Refer,$4,$6,make_pad(Comment,$7,NULL,NULL)); }
		;
oneline_block
		: LB RB_END
		{ $$ = make_pad(Box,get_headsp($2),NULL,NULL); }
		| LB RB_END comment
		{ $$ = make_pad(Box,get_headsp($2),make_pad(Comment,$3,NULL,NULL),NULL); }
		| LB string RB_END
		{ $$ = make_pad(Box,$2,NULL,NULL); }
		| LB string RB_END comment
		{ $$ = make_pad(Box,$2,make_pad(Comment,$4,NULL,NULL),NULL); }
		;
block
		: LB_END block_strings RB_END
		{ current = pop(); $$ = make_pad(Block,NULL,$2,NULL); }
		| LB_END block_strings RB_END comment
		{ current = pop();
		  $$ = make_pad(Block,NULL,$2,make_pad(Comment,$4,NULL,NULL)); }
		;
block_strings
		:
		{ push(current); $$ = NULL; }
		| real_block_strings
		{ $$ = $1; }
		;
real_block_strings
		: block_string
		{ push(current); current = $1; $$ = current; }
		| real_block_strings block_string
		{ link_pad($2); $$ = $1; }
		;
block_string
		: RET
		{ $$ = make_pad(Empty,NULL,NULL,NULL); }
		| string end
		{ $$ = make_pad(Str,joint(2,$1,$2),NULL,NULL); }
		| one_symbol_line
		{ $$ = make_pad(Str,$1,NULL,NULL); }
		| SHORT_LABEL
		{ $$ = make_pad(Str,rm_headsp(rm_tailsp(get_termstr($1))),NULL,NULL); }
		| DEFAULT_LABEL
		{ $$ = make_pad(Str,rm_headsp(rm_tailsp(get_termstr($1))),NULL,NULL); }
		;
normal_line
		: real_normal_line end
		{ $$ = make_pad(Box,joint(2,$1,$2),NULL,NULL); }
		| real_normal_line end comment
		{ $$ = make_pad(Box,joint(2,$1,$2),make_pad(Comment,$3,NULL,NULL),NULL); }
		| one_symbol_line
		{ $$ = make_pad(Box,$1,NULL,NULL); }
		| one_symbol_line comment
		{ $$ = make_pad(Box,$1,make_pad(Comment,$2,NULL,NULL),NULL); }
		;
real_normal_line
		: command_char
		{ $$ = $1; }
		| real_normal_line all_char
		{ $$ = joint(2,$1,$2); }
		| real_normal_line SP all_char
		{ $$ = joint(3,$1,$2,$3); }
		;
message
		: message_chars MESSAGE_END
		{ $$ = joint(2,$1,$2); }
		;
message_chars
		:				{ $$ = NULL; }
		| message_chars MESSAGE		{ $$ = joint(2,$1,$2); }
		;
comment
		: COMMENT
		{ $$ = $1; }
		| comment COMMENT
		{ $$ = joint(2,$1,$2); }
		;
string
		: real_string
		{ $$ = $1; }
		| SP string
		{ $$ = $2; }
		;
real_string
		: all_char
		{ $$ = $1; }
		| real_string all_char
		{ $$ = joint(2,$1,$2); }
		| real_string SP all_char
		{ $$ = joint(3,$1,$2,$3); }
		;
one_symbol_line
		: one_symbol_end
		{ $$ = rm_headsp($1); }
		;
command_char
		: CHAR			{ $$ = $1; }
		| LP			{ $$ = $1; }
		| RP			{ $$ = $1; }
		| RB			{ $$ = $1; }
		;
char
		: command_char		{ $$ = $1; }
		| FOR			{ $$ = $1; }
		| WHILE			{ $$ = $1; }
		| DO			{ $$ = $1; }
		| IF			{ $$ = $1; }
		| ELSE			{ $$ = $1; }
		| SWITCH		{ $$ = $1; }
		| CASE			{ $$ = $1; }
		| DEFAULT		{ $$ = $1; }
		| LABEL			{ $$ = $1; }
		| REFER			{ $$ = $1; }
		| TRY			{ $$ = $1; }
		| CATCH			{ $$ = $1; }
		;
all_char
		: char			{ $$ = $1; }
		| LB			{ $$ = $1; }
		| RB_ELSE		{ $$ = $1; }
		| RB_WHILE		{ $$ = $1; }
		;
sp
		: 
		| SP			{ free($1); }
		;
end
		: RET			{ $$ = NULL; }
		| RB_END		{ $$ = get_termstr($1); }
		| LB_END		{ $$ = get_termstr($1); }
		| one_symbol_end	{ $$ = $1; }
		;
one_symbol_end
		: RP_END		{ $$ = get_termstr($1); }
		| RPLB_END		{ $$ = get_termstr($1); }
		| COLON_END		{ $$ = get_termstr($1); }
		;
%%

#include "lex.yy.c"

PAD *make_pad(TYPE type,char *str,PAD *child,PAD *depth)
/* create a node on a tree structure */
{
    PAD *p;

    p = (PAD *)malloc(sizeof(PAD));
    p->type = type;
    p->str = str;
    p->child = child;
    p->depth = depth;
    p->next = NULL;

    return p;
}

void link_pad(PAD *p)
/* create link to below */
{
    current->next = p;
    current = p;
}

void dup_pad(PAD *p)
/* duplicate a node */
{
    current->depth = p;
    current = p;
}

char *get_termstr(char *s)
/* get string from terminator ("---\n") */
{
    int i,l;
    char *p;

    if(s == NULL)
	return NULL;

    l = strlen(s);
    for(i = l - 2 ; *(s+i) == ' '  ||  *(s+i) == '\t' ; i++)
	;

    l = i + 1;
    p = (char *)malloc((l+2)*sizeof(char));
    for(i = 0 ; i < l ; i++)
	*(p+i) = *(s+i);
    *(p+l) = '\0';

    free(s);
    return p;
}

char *get_until(char *s,char c,int flag)
/* get string that is before designated character */
{
    int i,l;
    char *p;

    if(s == NULL)
	return NULL;

    for(l = 0 ; *(s+l) != c ; l++)
	;
    if(flag == 1)
	l++;

    p = (char *)malloc((l+2)*sizeof(char));
    for(i = 0 ; i < l ; i++)
	*(p+i) = *(s+i);
    *(p+l) = '\0';

    free(s);
    return p;
}

char *get_headsp(char *s)
/* get spaces at head */
{
    int i,l;
    char *p;

    if(s == NULL)
	return NULL;

    for(l = 0 ; isspace(*(s+l)) ; l++)
	;

    p = (char *)malloc((l+2)*sizeof(char));
    for(i = 0 ; i < l ; i++)
	*(p+i) = ' ';
    *(p+l) = '\0';

    free(s);
    return p;
}

char *rm_headsp(char *s)
/* remove spaces at head */
{
    int i,l;
    char *p;

    if(s == NULL)
	return NULL;

    for(i = 0 ; isspace(*(s+i)) ; i++)
	;

    l = strlen(&s[i]);
    p = (char *)malloc((l+2)*sizeof(char));
    strcpy(p,&s[i]);

    free(s);
    return p;
}

char *rm_tailsp(char *s)
/* remove spaces at tail */
{
    int i;

    if(s == NULL)
	return NULL;

    for(i = strlen(s) - 1 ; i >= 0 && isspace(*(s+i)) ; i--)
	;
    s[i+1] = '\0';
    return s;
}

char *complete_label(char *str)
/* complete branch label in switch statement */
{
    char *p,*s;
    int l;

    p = pop_sw();

    if(p == NULL)
	s = str;
    else{
	l = strlen(p)+1;
	s = (char *)malloc((l+2)*sizeof(char));
	strcpy(s,p);
	strcat(s,"=");
	s = joint(2,s,str);
    }

    push_sw(p);
    return s;
}

void push_sw(void *ptr)
{
    STACK *tmp;

    tmp = (STACK *)malloc(sizeof(STACK));
    tmp->ptr = ptr;
    tmp->prev = sw;
    sw = tmp;
}

void *pop_sw(void)
{
    void *ptr;
    STACK *s;

    ptr = sw->ptr;
    s = sw;
    sw = sw->prev;
    free(s);
    return ptr;
}
