/*                              */
/*  pad2ps.c -- main routine    */
/*                              */
/*  pad2ps  by S. Yoshida       */
/*                              */

#include "config.h"

#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#include "pad2ps.h"

bool quiet = Off;		/* -quiet option */
#ifdef DEBUG
bool debug = Off;		/* -debug option */
#endif

bool eps_flag = Off;		/* pad2eps? */
bool latex_command_flag = Off;	/* treat string as LaTeX command? */

char *title = NULL;		/* overall title */
bool titlebox = Off;		/* whether to draw each PAD title in box */
bool page_print = Off;		/* whether to print page number */
int page = 0;			/* <starting page>-1 */
int figure = EOF;		/* figure number */
TYPE figure_locate = Top;	/* figure position */
bool figure_locate_flag = Off;	/* whether figure_locate is being set */
bool centering = On;		/* whether to centerize */
bool poster = Off;		/* poster */
bool slice = Off;		/* whether to draw each PAD from the head of page */

int tab_width = 4;		/* tab width */

TYPE locate_type = Middle;	/* PAD location format */

bool pageframe = Off;		/* whether to draw page frame */
bool padframe = On;		/* whether to draw PAD frame */

int page_top = PAGE_TOP;		/* page top */
int page_bottom = PAGE_BOTTOM;		/* page bottom */
int page_left = PAGE_LEFT;		/* page left */
int page_right = PAGE_RIGHT;		/* page right */
int page_width = PAGE_RIGHT-PAGE_LEFT;	/* page width */
int x_center = 0;		/* page center (when centering is 0n, this value is 0) */
int right_limit = PAGE_RIGHT;		/* right limit */
int left_limit = PAGE_LEFT;		/* left limit */

int max_length = INFINITY;		/* maximum length of PAD string */
int min_length = 0;			/* minumum length of PAD string */

int vspace = VSPACE;			/* vertical spacing between boxes */
int hspace = HSPACE;			/* horizontal spacing between boxes */
int box_vspace = BOX_VSPACE;		/* vertical space of normal box */
int if_vspace = IF_VSPACE;		/* vertical sapce of "if" box*/
int box_hspace = BOX_HSPACE;		/* horizontal space of normal box */
int if_rightspace = IF_RIGHTSPACE;	/* space of "if" box at right side */
int sideline_space = SIDELINE_SPACE;	/* spacing between horizontal sidelines (used in for statements and others) */
int box_height = BOX_VSPACE*2+FONTSIZE;	/* box height */
int if_height = IF_VSPACE*2+FONTSIZE;	/* "if" box height */
int pad_vspace = PAD_VSPACE;		/* vertical spacing between PADs */
int padframe_vspace = PADFRAME_VSPACE;	/* vertical spacing between PAD and its frame */
int padframe_hspace = PADFRAME_HSPACE;	/* horizontal spacing between PAD and its frame */

int linewidth = LINEWIDTH;		/* thickness of line */
int padlinewidth = PADLINEWIDTH;	/* thickness of PAD figure */
int framelinewidth = FRAMELINEWIDTH;	/* thickness of PAD frame */

int fontsize = FONTSIZE;		/* font size (height of characters) */
int fontwidth = FONTWIDTH;		/* width of font (alphabets) */
int jfontwidth = JFONTWIDTH;		/* width of font (japanese characters) */
double title_rate = TITLE_RATE;		/* magnification rate of title */
double padtitle_rate = PADTITLE_RATE;	/* magnification rate of PAD title */
double figure_rate = FIGURE_RATE;	/* magnification rate of PAD figure */

char font[] = "Courier";			/* font */
char title_font[] = "Courier-Bold";		/* title font */
#ifdef JAPANESE
char jfont[] = "Ryumin-Light-EUC-H";		/* Japanese font */
char title_jfont[] = "GothicBBB-Medium-EUC-H";	/* Japanese font for title */
#endif

double vrate = 1.0;		/* magnification rate (vertical) */
double hrate = 1.0;		/* magnification rate (horizontal) */

double proper_hrate = 99999.9;	/* appropriate hrate value */

int first_x = 0;		/* the starting x-coordinate */
int next_y = 10;		/* the following PAD's y-coordinate */

int most_right = 0;		/* the rightmost PAD's x-coordinate */
int most_left = PAGE_RIGHT;	/* the leftmost PAD's x-coordinate */

PAD *root = NULL;		/* root of the tree structure */
PAD *current = NULL;		/* current position of the tree structure */
BOX *first = NULL;		/* beginning of the list */
BOX *lp = NULL;			/* current position of the list */

bool print_warning = Off;	/* whether to print warning */
int line_count = 1;		/* line in process */

FILE *input_stream;		/* input file */
FILE *fp = stdout;		/* output file */
char outfile[BUFSIZ] = {'\0'};	/* output file name */
char outdir[BUFSIZ] = {'\0'};	/* directory to put output file */


#ifdef JAPANESE
static char kconv[BUFSIZ] = "nkf -e";	/* KANJI converter */
static bool noconv = Off;		/* -noconv */
#endif
static bool nohelp = Off;		/* -nohelp */
static bool parameter_flag = Off;	/* -P */
static bool stdin_flag = Off;		/* -stdin */

static char thiscmd[BUFSIZ] = {'\0'};	/* executed command name */
static char filename[BUFSIZ] = {'\0'};	/* input file name */
static char command[BUFSIZ];		/* command */
static char paramfile[BUFSIZ];		/* file name for handling parameters */


static void pad_parser(void);		/* call PAD parser */
static void parameter(void);		/* parameter set in -P */

static void option(int,char **);	/* handle options */
static void print_help(void);		/* print help messages */

#if defined(HAVE_ATEXIT)
static void at_exit(void);		/* handle termination process */
#elif defined(HAVE_ON_EXIT)
static void at_exit(int,void *);	/* handle termination process */
#endif

int main(int argc,char **argv)
{
    char *env;

    /* handle exit() */
#if defined(HAVE_ATEXIT)
    atexit(at_exit);
#elif defined(HAVE_ON_EXIT)
    on_exit(at_exit,NULL);
#endif

    /* file name for handling parameters */
#ifdef TMP
    strcpy(paramfile,TMP);
    strcat(paramfile,"/");
#else
    strcpy(paramfile,"/tmp/");
#endif
    if((env = getenv("TMP")) != NULL  &&  strlen(env) > 0){
	/* get the environment variable TMP */
	strcpy(paramfile,env);
	strcat(paramfile,"/");
    }
    strcat(paramfile,"pad2ps-XXXXXX");

    option(argc,argv);			/* handle options */

    if(quiet == Off)
	fprintf(stderr,"\n%s %s  by %s\n\n",thiscmd,VERSION,AUTHOR);

    pad_parser();			/* call PAD parser */
    parameter();			/* parameter set in -P */

    if(outfile[0] != '\0'  &&  outdir[0] != '\0'){	/* directory for output file */
	char tmp[BUFSIZ];
	strcpy(tmp,outfile);
	sprintf(outfile,"%s/%s",outdir,tmp);
    }

    if(strcmp(thiscmd,"pad2tex") == 0)
	tex();
    else{
	fold();
	locate();
	sort();
	if(strcmp(thiscmd,"pad2eps") == 0)
	    eps_flag = On;
	ps();
    }

    if(quiet == Off)
	fprintf(stderr,"\n%d errors, %d warnings.\n",error_count,warning_count);

    return 0;
}

void pad_parser(void)
/* call PAD parser */
{
    if(stdin_flag == On)
	filename[0] = '\0';

#ifdef JAPANESE
    if(noconv == Off)
	sprintf(command,"cat %s | %s",filename,kconv);
    else
#endif
	sprintf(command,"cat %s",filename);

    input_stream = popen(command,"r");

    parser();

    pclose(input_stream);
}

void parameter(void)
/* parameter set in -P */
{
    char tmp[BUFSIZ];

    if(parameter_flag == On){
#ifdef JAPANESE
	if(noconv == Off)
	    sprintf(command,"cat %s | %s",paramfile,kconv);
	else
#endif
	    sprintf(command,"cat %s",paramfile);

	input_stream = popen(command,"r");
	parser();
	pclose(input_stream);

	remove(paramfile);
    }
}

void option(int argc,char **argv)
/* handle options */
{
    int i,l,p = -1;

    for(i = 0 ; argv[0][i] != '\0' ; i++){	/* get executed command name */
	if(argv[0][i] == '/')
	    p = i;
    }
    strcpy(thiscmd,&argv[0][p+1]);

    for(i = 1 ; i < argc ; i++){
	if(argv[i][0] != '-'){
	    if(filename[0] == '\0'){		/* input file name */
		strcpy(filename,argv[i]);
		l = strlen(argv[i]);
		title = (char *)malloc((l+2) * sizeof(char));
		strcpy(title,argv[i]);
	    }else if(outfile[0] == '\0')	/* output file name */
		strcpy(outfile,argv[i]);
	    else if(nohelp == Off){
		print_help();
		exit(1);
	    }
	}else{				/* options */
	    switch(argv[i][1]){
		case 'd':
		    if(strcmp(argv[i],"-dir") == 0){
			strcpy(outdir,argv[++i]);
#ifdef DEBUG
		    }else if(strcmp(argv[i],"-debug") == 0){
			debug = On;
#endif
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		case 'h':
		    if(strcmp(argv[i],"-help") == 0){
			print_help();
			exit(0);
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
#ifdef JAPANESE
		case 'k':
		    if(strcmp(argv[i],"-kconv") == 0  &&  i + 1 < argc){
			strcpy(kconv,argv[++i]);
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
#endif
		case 'n':
		    if(strcmp(argv[i],"-nohelp") == 0){
			nohelp = On;
#ifdef JAPANESE
		    }else if(strcmp(argv[i],"-noconv") == 0){
			noconv = On;
#endif
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		case 'P':
		    if(strcmp(argv[i],"-P") == 0){
			FILE *fp;

			parameter_flag = On;
			mktemp(paramfile);
			fp = fopen(paramfile,"w");
			i++;
			while(i < argc  &&  argv[i][0] != '-'){
			    fprintf(fp,"/* pad2ps: %s */\n",argv[i]);
			    i++;
			}
			i--;
			fclose(fp);
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		case 'q':
		    if(strcmp(argv[i],"-quiet") == 0){
			quiet = On;
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		case 's':
		    if(strcmp(argv[i],"-stdin") == 0){
			stdin_flag = On;
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		case 't':
		    if(strcmp(argv[i],"-tab") == 0){
			i++;
			tab_width = atoi(argv[i]);
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		case 'w':
		    if(strcmp(argv[i],"-warn") == 0){
			print_warning = On;
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		default:
		    if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
	    }
	}
    }
}

void print_help(void)
{
    fprintf(stderr,"\n%s %s  by %s\n\n",thiscmd,VERSION,AUTHOR);
    fprintf(stderr,"usage: %s [inputfile [outputfile]] [options]\n",thiscmd);
#ifdef DEBUG
    fputs("-debug                  Print parser debug message.\n",stderr);
#endif
    fputs("-dir directory          Directory to output files.\n",stderr);
    fputs("-help                   Print help message.\n",stderr);
#ifdef JAPANESE
    fputs("-kconv converter        Kanji code converter [default: \"nkf -e\"].\n",stderr);
    fputs("-noconv                 Do not convert kanji code.\n",stderr);
#endif
    fputs("-nohelp                 Do not print help message.\n",stderr);
    fputs("-P parameters ...       Set parameters. Ex: -P page=1 figure=1\n",stderr);
    fputs("-quiet                  Do not print messages.\n",stderr);
    fputs("-stdin                  Input from stdin.\n",stderr);
    fputs("-tab width              Set TAB width [default: 4].\n",stderr);
    fputs("-warn                   Print warnings.\n",stderr);
}


#if defined(HAVE_ATEXIT) || defined(HAVE_ON_EXIT)
#if defined(HAVE_ATEXIT)
void at_exit(void)
#elif defined(HAVE_ON_EXIT)
void at_exit(int val, void *p)
#endif
/* handle termination process */
{
    remove(paramfile);
}
#endif
