/*                                  */
/*  main.c -- overall main routine  */
/*                                  */
/*  pad2ps  by S. Yoshida           */
/*                                  */

#include "config.h"

#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#include "main.h"

#ifdef DEBUG
static bool debug = Off;		/* -debug */
#endif
static bool ifdef = Off;		/* -ifdef */
#ifdef JAPANESE
static char kconv[BUFSIZ] = "nkf -e";	/* KANJI converter */
static bool noconv = Off;		/* -noconv */
#endif
static bool nohelp = Off;		/* -nohelp */
static bool quiet = Off;		/* -quiet */
static bool stdin_flag = Off;		/* -stdin */

static SRC source_type = Default;		/* type of source file */

static bool foo2pad_flag = Off;			/* whether foo2pad is executed */

static char thiscmd[BUFSIZ] = {'\0'};	/* executed command name */
static char filename[BUFSIZ] = {'\0'};	/* input file name */
static char outfile[BUFSIZ] = {'\0'};	/* output file name */
static char foo2padfile[BUFSIZ];	/* name of file to preserve output of foo2pad */


static void suffix(void);		/* select parser from suffix */
static void translator(int,char **);	/* execute foo2pad */
static void pad2ps(int,char **);	/* execute pad2p */

static void option(int,char **);		/* handle options */
static void mkargs(int,char **,char *,SRC);	/* create option strings */
static void print_help(void);			/* print help messages */

#if defined(HAVE_ATEXIT)
static void at_exit(void);		/* handle termination process */
#elif defined(HAVE_ON_EXIT)
static void at_exit(int,void *);	/* handle termination process */
#endif

int main(int argc,char **argv)
{
    char *env;

    /* handle exit() */
#if defined(HAVE_ATEXIT)
    atexit(at_exit);
#elif defined(HAVE_ON_EXIT)
    on_exit(at_exit,NULL);
#endif

    /* name of file to preserve output of foo2pad */
#ifdef TMP
    strcpy(foo2padfile,TMP);
    strcat(foo2padfile,"/");
#else
    strcpy(foo2padfile,"/tmp/");
#endif
    if((env = getenv("TMP")) != NULL  &&  strlen(env) > 0){
	/* get the environment variable TMP */
	strcpy(foo2padfile,env);
	strcat(foo2padfile,"/");
    }
    strcat(foo2padfile,"foo2pad-XXXXXX");

    option(argc,argv);			/* handle options */

    suffix();				/* select parser from suffix */
    translator(argc,argv);		/* execute foo2pad */
    pad2ps(argc,argv);			/* execute pad2ps */

    return 0;
}

void suffix(void)
/* select parser from suffix */
{
    int l;

    if(stdin_flag == On){
	if(filename[0] != '\0'){
	    strcpy(outfile,filename);
	    filename[0] = '\0';
	}
    }

    if(filename[0] != '\0'  &&  source_type == Default){
	l = strlen(filename);

	if(l >= 4  &&  strcmp(&filename[l-4],".awk") == 0)
	    source_type = Awk;
	if((l >= 2  &&  strcmp(&filename[l-2],".c") == 0)  ||
	   (l >= 2  &&  strcmp(&filename[l-2],".C") == 0)  ||
	   (l >= 3  &&  strcmp(&filename[l-3],".cc") == 0)  ||
	   (l >= 4  &&  strcmp(&filename[l-4],".cpp") == 0))
	    source_type = C;
	if(l >= 4  &&  strcmp(&filename[l-4],".csh") == 0)
	    source_type = Csh;
	if(l >= 5  &&  strcmp(&filename[l-5],".java") == 0)
	    source_type = Java;
	if(l >= 3  &&  strcmp(&filename[l-3],".sh") == 0)
	    source_type = Sh;
    }
}

void translator(int argc,char **argv)
/* execute foo2pad */
{
    char tmp[BUFSIZ];
    char lpath[BUFSIZ] = {'\0'};
    char module[BUFSIZ];
    char arguments[BUFSIZ] = {'\0'};
    char *env;
    int exit_code;
    FILE *chk;

#ifdef LPATH
    strcpy(lpath,LPATH);
    strcat(lpath,"/");
#endif

    if((env = getenv("PAD2PSLIB")) != NULL  &&  strlen(env) > 0){
	/* get the environment variable PAD2PSLIB */
	strcpy(lpath,env);
	strcat(lpath,"/");
    }

    switch(source_type){
	case Awk:
	    strcpy(module,"awk2pad");
	    break;
	case C:
	    strcpy(module,"c2pad");
	    break;
	case Csh:
	    strcpy(module,"csh2pad");
	    break;
	case Java:
	    strcpy(module,"java2pad");
	    break;
	case Sh:
	    strcpy(module,"sh2pad");
	    break;
	case Pad:
	case Default:
	    return;
    }

    sprintf(tmp,"%s%s",lpath,module);
    chk = fopen(tmp,"r");
    if(chk == NULL)		/* modules not in lpath */
	lpath[0] = '\0';
    fclose(chk);

    mkargs(argc,argv,arguments,source_type);

    if(source_type == C  &&  ifdef == On){	/* -ifdef */
	sprintf(tmp,"%sc2pad -nohelp %s",lpath,arguments);
	exit_code = system(tmp);
	exit(exit_code);
    }

    mktemp(foo2padfile);
    sprintf(tmp,"%s%s -nohelp -quiet %s > %s",lpath,module,arguments,foo2padfile);
    exit_code = system(tmp);
 
    foo2pad_flag = On;

    if(exit_code != 0){
	remove(foo2padfile);
	exit(exit_code);
    }
}

void pad2ps(int argc,char **argv)
/* execute pad2ps */
{
    char tmp[BUFSIZ];
    char lpath[BUFSIZ] = {'\0'};
    char arguments[BUFSIZ] = {'\0'};
    char dummy_inputfile[BUFSIZ] = {'\0'};
    char *env;
    int exit_code;
    FILE *chk;

#ifdef LPATH
    strcpy(lpath,LPATH);
    strcat(lpath,"/");
#endif

    if((env = getenv("PAD2PSLIB")) != NULL  &&  strlen(env) > 0){
	/* get the environment variable PAD2PSLIB */
	strcpy(lpath,env);
	strcat(lpath,"/");
    }

    sprintf(tmp,"%s%s",lpath,thiscmd);
    chk = fopen(tmp,"r");
    if(chk == NULL){		/* modules not in lpath */
	fclose(chk);
	fprintf(stderr,"No %s module.\n",thiscmd);
	exit(1);
    }
    fclose(chk);

    if(stdin_flag == On  &&  outfile[0] != '\0')
	strcpy(dummy_inputfile,outfile);

    mkargs(argc,argv,arguments,Pad);
    if(source_type != Pad  &&  source_type != Default)
	sprintf(tmp,"cat %s | %s%s -nohelp -stdin %s %s",foo2padfile,lpath,thiscmd,dummy_inputfile,arguments);
    else
	sprintf(tmp,"%s%s -nohelp %s %s",lpath,thiscmd,dummy_inputfile,arguments);
    exit_code = system(tmp);

    if(source_type != Pad  &&  source_type != Default)
	remove(foo2padfile);

    if(exit_code != 0)
	exit(exit_code);
}

void option(int argc,char **argv)
/* handle options */
{
    int i,p = -1;

    for(i = 0 ; argv[0][i] != '\0' ; i++){	/* get executed command name */
	if(argv[0][i] == '/')
	    p = i;
    }
    strcpy(thiscmd,&argv[0][p+1]);

    for(i = 1 ; i < argc ; i++){
	if(argv[i][0] != '-'){
	    if(filename[0] == '\0'){		/* input file name */
		strcpy(filename,argv[i]);
	    }else if(outfile[0] == '\0')	/* output file name */
		strcpy(outfile,argv[i]);
	    else if(nohelp == Off){
		print_help();
		exit(1);
	    }
	}else{				/* options */
	    switch(argv[i][1]){
		case 'a':
		    if(strcmp(argv[i],"-awk") == 0){
			source_type = Awk;
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		case 'c':
		    if(strcmp(argv[i],"-c") == 0){
			source_type = C;
		    }else if(strcmp(argv[i],"-c++") == 0){
			source_type = C;
		    }else if(strcmp(argv[i],"-csh") == 0){
			source_type = Csh;
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		case 'D':
		    if(strcmp(argv[i],"-D") == 0){
			i++;
			while(i < argc  &&  argv[i][0] != '-')
			    i++;
			i--;
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		case 'd':
		    if(strcmp(argv[i],"-dir") == 0){
			i++;
#ifdef DEBUG
		    }else if(strcmp(argv[i],"-debug") == 0){
			debug = On;
#endif
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		case 'h':
		    if(strcmp(argv[i],"-help") == 0){
			print_help();
			exit(0);
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		case 'i':
		    if(strcmp(argv[i],"-ifdef") == 0){
			ifdef = On;
		    }else if(strcmp(argv[i],"-inline") == 0){
			break;
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		case 'j':
		    if(strcmp(argv[i],"-java") == 0){
			source_type = Java;
		    }
		    break;
#ifdef JAPANESE
		case 'k':
		    if(strcmp(argv[i],"-kconv") == 0  &&  i + 1 < argc){
			strcpy(kconv,argv[++i]);
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
#endif
		case 'n':
		    if(strcmp(argv[i],"-nohelp") == 0){
			nohelp = On;
		    }else if(strcmp(argv[i],"-noopt") == 0){
			break;
#ifdef JAPANESE
		    }else if(strcmp(argv[i],"-noconv") == 0){
			noconv = On;
#endif
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		case 'O':
		    if(strcmp(argv[i],"-O") == 0){
			break;
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		case 'P':
		    if(strcmp(argv[i],"-P") == 0){
			i++;
			while(i < argc  &&  argv[i][0] != '-')
			    i++;
			i--;
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		case 'p':
		    if(strcmp(argv[i],"-pad") == 0){
			source_type = Pad;
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		case 'q':
		    if(strcmp(argv[i],"-quiet") == 0){
			quiet = On;
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		case 's':
		    if(strcmp(argv[i],"-semicolon") == 0){
			break;
		    }else if(strcmp(argv[i],"-stdin") == 0){
			stdin_flag = On;
		    }else if(strcmp(argv[i],"-sh") == 0){
			source_type = Sh;
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		case 't':
		    if(strcmp(argv[i],"-tab") == 0){
			i++;
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		case 'w':
		    if(strcmp(argv[i],"-warn") == 0){
			break;
		    }else if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
		default:
		    if(nohelp == Off){
			print_help();
			exit(1);
		    }
		    break;
	    }
	}
    }
}

void mkargs(int argc,char **argv,char *args,SRC src_type)
/* create option strings */
{
    int i;

    args[0] = '\0';

    for(i = 1 ; i < argc ; i++){
	if(argv[i][0] != '-'){		/* file name */
	    strcat(args," ");
	    strcat(args,argv[i]);
	}else{				/* options */
	    switch(argv[i][1]){
		case 'D':
		    if(strcmp(argv[i],"-D") == 0){
			if(src_type == C)
			    strcat(args," -D");
			i++;
			while(i < argc  &&  argv[i][0] != '-'){
			    if(src_type == C){
				strcat(args," ");
				strcat(args,argv[i]);
			    }
			    i++;
			}
			if(src_type == C)
			    strcat(args," -dummy");
			i--;
		    }
		    break;
		case 'd':
		    if(strcmp(argv[i],"-dir") == 0){
			i++;
			if(src_type == Pad  ||  src_type == Default){
			    strcat(args," -dir ");
			    strcat(args,argv[i]);
			}
#ifdef DEBUG
		    }else if(strcmp(argv[i],"-debug") == 0){
			strcat(args," -debug");
#endif
		    }
		    break;
		case 'i':
		    if(strcmp(argv[i],"-ifdef") == 0){
			if(src_type == C)
			    strcat(args," -ifdef");
		    }else  if(strcmp(argv[i],"-inline") == 0){
			if(src_type == C)
			    strcat(args," -inline");
		    }
		    break;
#ifdef JAPANESE
		case 'k':
		    if(strcmp(argv[i],"-kconv") == 0  &&  i + 1 < argc){
			strcat(args," -kconv ");
			strcat(args,argv[++i]);
		    }
		    break;
#endif
		case 'n':
		    if(strcmp(argv[i],"-noopt") == 0){
			if(src_type == C  ||  src_type == Java)
			    strcat(args," -noopt");
#ifdef JAPANESE
		    }else if(strcmp(argv[i],"-noconv") == 0){
			strcat(args," -noconv");
#endif
		    }
		    break;
		case 'O':
		    if(strcmp(argv[i],"-O") == 0){
			if(src_type == C  ||  src_type == Java)
			    strcat(args," -O");
		    }
		    break;
		case 'P':
		    if(strcmp(argv[i],"-P") == 0){
			if(src_type == Pad  ||  src_type == Default)
			    strcat(args," -P");
			i++;
			while(i < argc  &&  argv[i][0] != '-'){
			    if(src_type == Pad  ||  src_type == Default){
				strcat(args," ");
				strcat(args,argv[i]);
			    }
			    i++;
			}
			if(src_type == Pad  ||  src_type == Default)
			    strcat(args," -dummy");
			i--;
		    }
		    break;
		case 'q':
		    if(strcmp(argv[i],"-quiet") == 0){
			strcat(args," -quiet");
		    }
		    break;
		case 's':
		    if(strcmp(argv[i],"-semicolon") == 0){
			if(src_type == C  ||  src_type == Java)
			    strcat(args," -semicolon");
		    }else if(strcmp(argv[i],"-stdin") == 0){
			strcat(args," -stdin");
		    }
		    break;
		case 't':
		    if(strcmp(argv[i],"-tab") == 0){
			i++;
			strcat(args," -tab ");
			strcat(args,argv[i]);
		    }
		    break;
		case 'w':
		    if(strcmp(argv[i],"-warn") == 0){
			strcat(args," -warn");
		    }
		    break;
	    }
	}
    }
}

void print_help(void)
{
    fprintf(stderr,"\n%s %s  by %s\n\n",thiscmd,VERSION,AUTHOR);
    fprintf(stderr,"usage: %s [inputfile [outputfile]] [options]\n",thiscmd);
    fputs("-awk                    Draw PAD of AWK program directly.\n",stderr);
    fputs("-c                      Draw PAD of C program directly.\n",stderr);
    fputs("-c++                    Draw PAD of C++ program directly.\n",stderr);
    fputs("-csh                    Draw PAD of C shell program directly.\n",stderr);
    fputs("-D symbols ...          Define symbols. Ex: -D DEBUG __linux__  <C,C++>\n",stderr);
#ifdef DEBUG
    fputs("-debug                  Print parser debug message.\n",stderr);
#endif
    fputs("-dir directory          Directory to output files.\n",stderr);
    fputs("-help                   Print help message.\n",stderr);
    fputs("-ifdef                  Print defined symbols.  <C,C++>\n",stderr);
    fputs("-inline                 Draw PAD of inline functions too.  <C++>\n",stderr);
    fputs("-java                   Draw PAD of Java program directly.\n",stderr);
#ifdef JAPANESE
    fputs("-kconv converter        Kanji code converter [default: \"nkf -e\"].\n",stderr);
    fputs("-noconv                 Do not convert kanji code.\n",stderr);
#endif
    fputs("-nohelp                 Do not print help message.\n",stderr);
    fputs("-noopt                  No optimizing.  <C,C++,Java>\n",stderr);
    fputs("-O                      Optimizing.  <C,C++,Java>\n",stderr);
    fputs("-P parameters ...       Set parameters. Ex: -P page=1 figure=1\n",stderr);
    fputs("-pad                    Draw PAD of PAD source (PADEL).\n",stderr);
    fputs("-quiet                  Do not print messages.\n",stderr);
    fputs("-semicolon              Print ';' at EOL.  <C,C++,Java>\n",stderr);
    fputs("-sh                     Draw PAD of Bourne shell program directly.\n",stderr);
    fputs("-stdin                  Input from stdin.\n",stderr);
    fputs("-tab width              Set TAB width [default: 4].\n",stderr);
    fputs("-warn                   Print warnings.\n",stderr);
}

#if defined(HAVE_ATEXIT) || defined(HAVE_ON_EXIT)
#if defined(HAVE_ATEXIT)
void at_exit(void)
#elif defined(HAVE_ON_EXIT)
void at_exit(int val, void *p)
#endif
/* hadle termination process */
{
    remove(foo2padfile);
}
#endif
