/*                                                       */
/*  parser.c -- Translate Java to PADEL and print result */
/*                                                       */
/*  java2pad  by S. Yoshida                              */
/*                                                       */

#include "config.h"

#include <string.h>
#ifdef HAVE_MALLOC_H
# include <malloc.h>
#endif
#include <ctype.h>
#include "java2pad.h"
#include "lib.h"

#define MAKE_STRING_WHILE(X)	{\
				    label_flag = On;\
				    set_str();\
				    get_token();\
				    while(X){\
					if(token == Ret){\
					    append(" ");\
					    *sp = '\0';\
					    do{\
						get_token();\
					    }while(token == Sp);\
					}else{\
					    if(token == Sp)\
						strcat(sp,word);\
					    else{\
						if(token == Question)\
						    label_flag = Off;\
						if(*sp != '\0')\
						    append(sp);\
						*sp = '\0';\
						append_str();\
					    }\
					    get_token();\
					}\
				    }\
				    if(*sp != '\0')\
					append(sp);\
				    *sp = '\0';\
				}

#define END_OF_LINE()		{\
				    while(token == Sp)\
					get_token();\
				    if(token == Ret)\
					get_token();\
				}

static char *str = NULL;
static int nest_level = 0;
static bool label_flag = On;
static bool in_class = Off;

static void function(void);
static void body(int);
static void command(void);
static void for_statement(void);
static void else_statement(bool);
static void do_statement(void);
static void try_statement(void);
static void condition(void);
static void paired_brace(void);

static void set_str(void);
static void append_str(void);
static void append(char *);
static void print_str(void);
static void print_hsp(void);

static int to_optimize(char *);

void parser(void)
/* output */
{
#ifdef DEBUG
    if(quiet == Off)
	fprintf(stderr,"Translating Java into PADEL starts...");
#endif

    get_token();

    while(token != Eof){
	while(token == Sp)
	    get_token();

	if(token == Ret){		/* empty line */
	    putchar('\n');
	    get_token();
	}else if(token != Eof)
	    function();			/* declaration, function */
    }

#ifdef DEBUG
    if(quiet == Off)
	fprintf(stderr,"done\n");
#endif
}

void function(void)
/* declaration, function */
{
    char sp[BUFSIZ] = {'\0'};
    bool ret_flag = Off;

    MAKE_STRING_WHILE(token != Lpar  &&  token != Lbrace  &&  token != Lbracket  &&  token != Semicolon  &&  token != Eof);

    if(token == Lpar){
	condition();			/* argument list */
	while(token == Sp  ||  token == Ret){
	    if(token == Sp)
		strcat(sp,word);
	    else if(token == Ret)
		ret_flag = On;
	    get_token();
	}

	if(token != Semicolon){
	    while(token != Lbrace)	/* type declaration of argument */
		get_token();

	    print_str();
	    if(ret_flag == On)
		printf("\n{\n");
	    else
		printf("%s\n{\n",sp);

	    nest_level++;
	    body(0);			/* function main */
	    nest_level--;

	    printf("}\n");

	    END_OF_LINE();
	    return;
	}
    }

    if(token == Lbrace){		/* struct, class, and others */
	if(in_class == Off){		/* class */
	    printf("// ");
	    print_str();
	    putchar('\n');

	    in_class = On;
	    get_token();
	    while(token != Rbrace  &&  token != Eof){
		while(token == Sp)
		    get_token();

		if(token == Ret){		/* empty line */
		    putchar('\n');
		    get_token();
		}else if(token != Eof)
		    function();			/* declaration, function */
	    }
	    in_class = Off;
	}else{
	    paired_brace();
	    while(token != Semicolon){
		if(token == Lbrace)
		    paired_brace();
		else if(token == Lpar)
		    condition();
		else
		    get_token();
	    }
	}
    }

    if(token == Lbracket){		/* array and others */
	get_token();
	while(token != Semicolon){
	    if(token == Lbrace)
		paired_brace();
	    else if(token == Lpar)
		condition();
	    else
		get_token();
	}
    }

    if(token == Semicolon){		/* global variable declaration and others */
	get_token();
	END_OF_LINE();
	if(str != NULL){
	    free(str);
	    str = NULL;
	}
	return;
    }
}

void body(int flag)
{
    if(flag != 1){
	get_token();
	END_OF_LINE();

	while(token == Sp)
	    get_token();
    }

    while(token != Rbrace  &&  token != Eof){
	switch(token){
	    case Ret:			/* empty line */
		putchar('\n');
		get_token();
		break;
	    case Lbrace:		/* emblaced by { } */
		body(0);
		END_OF_LINE();
		break;
	    default:
		command();
	}

	while(token == Sp)
	    get_token();
    }

    get_token();
}

void command(void)
{
    char sp[BUFSIZ] = {'\0'};
    char buf[BUFSIZ];
    char *s;
    int count;

    switch(token){
	case For:
	case While:
	case If:
	case Switch:
	    for_statement();
	    break;
	case Do:
	    do_statement();
	    break;
	case Try:
	    try_statement();
	    break;
	case Case:
	case Default:
	    MAKE_STRING_WHILE(token != Colon);
	    nest_level--;
	    print_hsp();
	    nest_level++;
	    print_str();
	    printf(":\n");
	    get_token();
	    while(token == Sp  ||  token == Ret)
		get_token();
	    break;
	case Semicolon:
	    print_hsp();
	    if(semicolon == On)
		printf(";\n");
	    else
		printf("{    }\n");
	    get_token();
	    END_OF_LINE();
	    break;
	case Break:
	case Continue:
	case Goto:
	    strcpy(buf,word);
	    get_token();
	    if (token == Sp  ||  token == Ret)
		MAKE_STRING_WHILE(token == Sp  ||  token == Ret);
	    if (token != Semicolon) {	/* reference exists */
		print_hsp();
		printf("refer (%s%s%s) {\n",str+1,word,str+1);
		nest_level++;
		print_hsp();
		puts(buf);
		nest_level--;
		print_hsp();
		puts("}");

		free(str);
		MAKE_STRING_WHILE(token != Semicolon  &&  token != Eof);
		free(str);
		str = NULL;
	    } else {			/* no reference */
		print_hsp();
		fputs(buf,stdout);
		if(str != NULL)
		    print_str();
		if(semicolon == On)
		    putchar(';');
		putchar('\n');
	    }

	    get_token();
	    END_OF_LINE();
	    break;
	case Return:
	default:
	    MAKE_STRING_WHILE(token != Semicolon  &&  token != Lbrace
			      &&  token != Colon);
	    if(token == Colon  &&  label_flag == On){	/* label */
		print_str();
		printf(":\n");
	    }else{
		if(token == Colon){		/* (cond ? T : F) form */
		    s = str;
		    MAKE_STRING_WHILE(token != Semicolon);
		    str = joint(2,s,str);
		}else if(token == Lbrace){		/* struct and others */
		    count = 1;
		    s = str;
		    while(token == Lbrace){
			do{
			    MAKE_STRING_WHILE(token != Rbrace  &&  token != Lbrace);
			    s = joint(2,s,str);
			    count += (token == Lbrace) ? 1 : -1;
			}while(count > 0);
			MAKE_STRING_WHILE(token != Semicolon  &&  token != Lbrace);
			s = joint(2,s,str);
		    }
		    str = s;
		}
		if(to_optimize(str)){
		    free(str);
		}else{
		    print_hsp();
		    print_str();
		    if(semicolon == On)
			putchar(';');
		    putchar('\n');
		}
	    }
	    get_token();
	    END_OF_LINE();
    }
}

void for_statement(void)
{
    char *old_str, sp[BUFSIZ] = {'\0'};
    char ret_buf[BUFSIZ] = {'\0'}, comm_buf[BUFSIZ] = {'\0'};
    bool ret_flag = Off;
    token_type type;
    int i,sp_len,ret_count = 0;
    int flag = 0;

    type = token;

    old_str = str;
    MAKE_STRING_WHILE(token != Lpar);
    if(elseif_flag == On)	/* create string following "else" */
	str = joint(2,old_str,str);
    condition();
    while(token == Sp  ||  token == Ret){
	if(token == Sp)
	    strcat(sp,word);
	else if(token == Ret)
	    ret_flag = On;
	get_token();
    }

    if(type == Switch){		/* part in between "switch(){" and first "case" */
	flag = 1;
	old_str = str;
	str = NULL;
	get_token();
	END_OF_LINE();
	while(token == Sp)
	    get_token();
	while(token != Case  &&  token != Default  &&  token != Rbrace  &&  token != Eof){
	    if(token == Ret){
		putchar('\n');
		get_token();
	    }else
		command();
	    while(token == Sp)
		get_token();
	}
	str = old_str;
    }

    if(elseif_flag == On)	/* spacing at the head of line in not required with "else if" */
	elseif_flag = Off;
    else
	print_hsp();

    print_str();

    if(flag == 1  ||  token == Lbrace){
	if(ret_flag == On){
	    putchar('\n');
	    print_hsp();
	    printf("{\n");
	}else
	    printf("%s{\n",sp);

	nest_level++;
	if(type == Switch)
	    nest_level++;
	body(flag);				/* repetition body */
	nest_level--;
	if(type == Switch)
	    nest_level--;

	if(type != If){
	    print_hsp();
	    printf("}\n");
	    END_OF_LINE();
	}else{
	    sp_len = nest_level * tab;
	    for(i = 0 ; i < sp_len ; i++)
		append(" ");
	    append("}");
	    print_str();
	}

	ret_flag = On;
    }else{
	putchar('\n');
	nest_level++;
	command();
	nest_level--;
	ret_flag = Off;
    }

    *sp = '\0';
    ret_count = 0;
    if(type == If){			/* whether "else" exists */
	no_print_comment_flag = On;
	while(token == Sp  ||  token == Ret  ||  token == Comment){
	    if(token == Sp){
		strcat(sp,word);
	    }else if(token == Ret){
		ret_count++;
		strcat(ret_buf,"\n");
	    }else if(token == Comment){
		strcat(ret_buf,word);
		strcat(comm_buf,word);
		ret_count++;
		if (word[strlen(word)-1] != '\n') {
		    strcat(ret_buf,"\n");
		    strcat(comm_buf,"\n");
		    get_token();
		    while (token == Sp)
			get_token();
		    if (token == Ret)
			get_token();
		    continue;
		}
	    }
	    get_token();
	}
	no_print_comment_flag = Off;
	if(token != Else){		/* if no "else" exists */
	    print_str();
	    if(ret_flag == On  &&  ret_count == 0)
		strcpy(ret_buf,"\n");
	    fputs(ret_buf,stdout);
	    return;
	}else{				/* if "else" exists */
	    sp_len = nest_level * tab;
	    if (*comm_buf != '\0') {
		if(ret_flag == On)
		    putchar('\n');
		fputs(comm_buf,stdout);
		ret_flag = Off;
	    } else {
		if(ret_flag == On){
		    if(ret_count > 0){
			putchar('\n');
			ret_flag = Off;
		    }else
			sp_len = strlen(sp);
		}
	    }

	    for(i = 0 ; i < sp_len ; i++)
		append(" ");
	    append("else");
	    else_statement(ret_flag);
	}
    }
}

void else_statement(bool old_ret_flag)
{
    char sp[BUFSIZ] = {'\0'};
    bool ret_flag = Off;
    bool mid_comm_flag = Off;

    no_print_comment_flag = On;
    get_token();
    while(token == Sp  ||  token == Ret  ||  token == Comment){
	if(token == Sp)
	    strcat(sp,word);
	else if(token == Ret)
	    ret_flag = On;
	else if(token == Comment){
	    mid_comm_flag = On;
	    if (old_ret_flag == On) {
		putchar('\n');
		old_ret_flag = Off;
	    }
	    if (word[strlen(word)-1] != '\n') {
		puts(word);
		get_token();
		while (token == Sp)
		    get_token();
		if (token == Ret)
		    get_token();
		continue;
	    }
	    printf("%s",word);
	}
	get_token();
    }
    no_print_comment_flag = Off;

    if(token == If){
	if(ret_flag == Off){		/* else if */
	    append(sp);
	    if (mid_comm_flag = Off)
	        elseif_flag = On;
	    command();
	}else{
	    print_str();
	    putchar('\n');
	    nest_level++;
	    command();
	    nest_level--;
	}
    }else{
	print_str();
	if(token == Lbrace){
	    if(ret_flag == On){
		putchar('\n');
		print_hsp();
		printf("{\n");
	    }else
		printf("%s{\n",sp);

	    nest_level++;
	    body(0);			/* repetition body */
	    nest_level--;

	    print_hsp();
	    printf("}\n");

	    END_OF_LINE();
	    return;
	}else{
	    putchar('\n');
	    nest_level++;
	    command();
	    nest_level--;
	}
    }
}

void do_statement(void)
{
    char sp[BUFSIZ] = {'\0'};
    bool ret_flag = Off;

    get_token();
    while(token == Sp  ||  token == Ret){
	if(token == Sp)
	    strcat(sp,word);
	else if(token == Ret)
	    ret_flag = On;
	get_token();
    }

    print_hsp();

    if(token == Lbrace){
	if(ret_flag == On)
	    printf("do\n{\n");
	else
	    printf("do%s{\n",sp);

	nest_level++;
	body(0);				/* repetition body */
	nest_level--;

	print_hsp();
	printf("}");
    }else{
	printf("do\n");
	nest_level++;
	command();
	nest_level--;
	while(token == Sp  ||  token == Ret)
	    get_token();
	print_hsp();
    }

    *sp = '\0';
    MAKE_STRING_WHILE(token != Lpar);
    condition();

    print_str();
    putchar('\n');

    while(token != Semicolon)
	get_token();
    get_token();
    END_OF_LINE();
}

void try_statement(void)
{
    char sp[BUFSIZ] = {'\0'};
    bool ret_flag = Off;

    get_token();
    while(token == Sp  ||  token == Ret){
	if(token == Sp)
	    strcat(sp,word);
	else if(token == Ret)
	    ret_flag = On;
	get_token();
    }

    print_hsp();

    if(token == Lbrace){
	if(ret_flag == On)
	    printf("try\n{\n");
	else
	    printf("try%s{\n",sp);

	nest_level++;
	body(0);				/* try body */
	nest_level--;

	print_hsp();
	printf("}");

	ret_flag = Off;
	*sp = '\0';
	while(token == Sp  ||  token == Ret){
	    if(token == Ret)
		ret_flag = On;
	    else
		strcat(sp,word);
	    get_token();
	}
	if(ret_flag == On){
	    putchar('\n');
	    print_hsp();
	}else
	    fputs(sp,stdout);
    }else{
	printf("try\n");
	nest_level++;
	command();
	nest_level--;
	while(token == Sp  ||  token == Ret)
	    get_token();
	print_hsp();
    }

    *sp = '\0';
    MAKE_STRING_WHILE(token != Lpar);
    condition();

    print_str();

    ret_flag = Off;
    *sp = '\0';
    while(token == Sp  ||  token == Ret){
	if(token == Sp)
	    strcat(sp,word);
	else if(token == Ret)
	    ret_flag = On;
	get_token();
    }

    if(token == Lbrace){
	if(ret_flag == On)
	    printf("\n{\n");
	else
	    printf("%s{\n",sp);

	nest_level++;
	body(0);				/* catch body */
	nest_level--;

	print_hsp();
	puts("}");
	get_token();
	END_OF_LINE();
    }else{
	putchar('\n');
	nest_level++;
	command();
	nest_level--;
	while(token == Sp  ||  token == Ret)
	    get_token();
    }
}

void condition(void)
{
    char sp[BUFSIZ] = {'\0'};

    append_str();
    get_token();

    while(token != Rpar){
	if(token == Lpar)
	    condition();
	else if(token == Ret){
	    append(" ");
	    *sp = '\0';
	    do{
		get_token();
	    }while(token == Sp);
	}else{
	    if(token == Sp)
		strcat(sp,word);
	    else{
		if(*sp != '\0')
		    append(sp);
		*sp = '\0';
		append_str();
	    }

	    get_token();
	}
    }

    if(*sp != '\0')
	append(sp);
    append_str();
    get_token();
}

void paired_brace(void)
{
    get_token();

    while(token != Rbrace){
	if(token == Lbrace)
	    paired_brace();
	else
	    get_token();
    }
    get_token();
}

void set_str(void)
{
    int l;

    l = strlen(word);
    str = (char *)malloc((l+2) * sizeof(char));
    strcpy(str,word);
}

void append_str(void)
{
    int l;
    char *s;

    l = strlen(word);
    s = (char *)malloc((l+2) * sizeof(char));
    strcpy(s,word);
    str = joint(2,str,s);
}

void append(char *p)
{
    int l;
    char *s;

    l = strlen(p);
    s = (char *)malloc((l+2) * sizeof(char));
    strcpy(s,p);
    str = joint(2,str,s);
}

void print_str(void)
{
    if(str != NULL){
	printf("%s",str);
	free(str);
	str = NULL;
    }
}

void print_hsp(void)
{
    int i;

    for(i = 0 ; i < nest_level * tab ; i++)
	putchar(' ');
}

int to_optimize(char *s)
/* return 1 if discarded by optimization; otherwise return 0 */
{
    int flag = 1;
    char *p;

    if(optimize == On){
	for(p = s ; *p != '\0' ; p++){
	    switch(*p){
		case '_':
		case ',':
		case ';':
		case '*':
		case '[':
		case ']':
		case ' ':
		case '\t':
		    break;
		default:
		    if( ! (isalnum(*p)))
			flag = 0;
	    }
	}
	return flag;
    }

    return 0;
}
