/*                                 */
/*  token.c -- get token           */
/*                                 */
/*  csh2pad  by S. Yoshida         */
/*                                 */

#include "config.h"

#include <string.h>
#ifdef HAVE_MALLOC_H
# include <malloc.h>
#endif
#include <ctype.h>
#include "csh2pad.h"
#include "lib.h"

static int ch = 0;
static int pch = 0;		/* read one character in ahead */
static int word_ptr = 0;	/* pointer to the end of word */
static bool newline = On;	/* whether at the head of line or not */
static bool comm_flag = Off;	/*  whether in comment or not */

static void set_ch(void);
static void append_ch(void);
static void get_ch(void);
static int p_getchar(FILE *);

void get_token(void)
/* get token */
{
    if(token == Eof)
	fatal_error("input file terminates unexpectedly.");

    if(ch == 0)
	get_ch();

    if(cat_flag > 0){
	if(ch == '\n')
	    token = Ret;
	else{
	    token = Str;
	    *word = ch;
	    *(word+1) = '\0';
	}
	get_ch();
	return;
    }

    switch(ch){
	case EOF:
	    token = Eof;
	    *word = '\0';
	    return;
	case '\n':
	    newline = On;
	    token = Ret;
	    *word = ' ';
	    *(word+1) = '\0';
	    get_ch();
	    cat_flag = strlen(key);
	    return;
	case ' ':
	case '\t':
	    token = Sp;
	    set_ch();
	    get_ch();
	    while(ch == ' '  ||  ch == '\t'){
		append_ch();
		get_ch();
	    }
	    return;
	case '#':
	    comm_flag = On;
	    set_ch();
	    get_ch();
	    while(ch != '\n'){
		append_ch();
		get_ch();
	    }

	    /* do not print at this place */
	    if (no_print_comment_flag == On) {
		char buf[BUFSIZ];
		strcpy(buf,"// ");
		strcat(buf,word);
		comm_flag = Off;

		if(word[strlen(word)-1] == '\\'  ||  newline == On){
		    set_ch();
		    get_ch();
		}else{
		    cat_flag = strlen(key);
		}

		newline = On;
		strcpy(word,buf);
		token = Comment;
		return;
	    }

	    if (elseif_flag == On) {
	        putchar('\n');
		elseif_flag = Off;
	    }
	    printf("// ");
	    puts(word);
	    comm_flag = Off;
	    set_ch();
	    get_ch();
	    if(word[strlen(word)-1] != '\\'  &&  newline != On){	/* insert return code if it is not a comment at the head of line */
		newline = On;
		token = Ret;
		cat_flag = strlen(key);
	    }else{
		newline = On;
		get_token();
	    }
	    return;
	case '"':
	    newline = Off;
	    set_ch();

	    do{
		get_ch();
		if(ch == '\n'){
		    ch = '\\'; append_ch();
		    ch = 'n'; append_ch();
		}else if(ch == '\\'){
		    append_ch();
		    get_ch(); append_ch();
		    ch = ' ';
		}else
		    append_ch();
	    }while(ch != '"');

	    get_ch();

	    token = Str;
	    return;
	case '\'':
	    newline = Off;
	    set_ch();

	    do{
		get_ch();
		if(ch == '\n'){
		    ch = '\\'; append_ch();
		    ch = 'n'; append_ch();
		}else if(ch == '\\'){
		    append_ch();
		    get_ch(); append_ch();
		    ch = ' ';
		}else
		    append_ch();
	    }while(ch != '\'');

	    get_ch();

	    token = Str;
	    return;
	case '`':
	    newline = Off;
	    set_ch();

	    do{
		get_ch();
		if(ch == '\n'){
		    ch = '\\'; append_ch();
		    ch = 'n'; append_ch();
		}else if(ch == '\\'){
		    append_ch();
		    get_ch(); append_ch();
		    ch = ' ';
		}else
		    append_ch();
	    }while(ch != '`');

	    get_ch();

	    token = Str;
	    return;
	case '&':
	    newline = Off;
	    set_ch();
	    get_ch();
	    if (ch == '&') {
		token = And2;
		append_ch();
		get_ch();
	    } else
		token = Str;
	    return;
	case '|':
	    newline = Off;
	    set_ch();
	    get_ch();
	    if (ch == '|') {
		token = Or2;
		append_ch();
		get_ch();
	    } else
		token = Str;
	    return;
	case '<':
	    newline = Off;
	    set_ch();
	    get_ch();
	    if (ch == '<') {
		token = LL;
		append_ch();
		get_ch();
	    } else
		token = Str;
	    return;
	case '\\':
	    newline = Off;
	    token = Str;
	    set_ch();
	    get_ch();
	    append_ch();
	    get_ch();
	    return;
	case '$':
	    newline = Off;
	    token = Str;
	    set_ch();
	    get_ch();
	    if(ch == '#'){
		append_ch();
		get_ch();
	    }
	    return;
	case '(':
	    newline = Off;
	    token = Lpar;
	    set_ch();
	    get_ch();
	    return;
	case ')':
	    newline = Off;
	    token = Rpar;
	    set_ch();
	    get_ch();
	    return;
	case ':':
	    newline = Off;
	    set_ch();
	    get_ch();
	    token = Colon;
	    return;
	case ';':
	    newline = Off;
	    token = Semicolon;
	    set_ch();
	    get_ch();
	    return;
	default:
	    if(isdigit(ch)){
		token = Num;
		set_ch();
		get_ch();
		while(isdigit(ch)){
		    append_ch();
		    get_ch();
		}
		newline = Off;
		return;
	    }

	    if(isalpha(ch)  ||  ch == '_'){
		token = Name;
		set_ch();
		get_ch();
		while(isalnum(ch)  ||  (char)ch == '_'){
		    append_ch();
		    get_ch();
		}

		if(strcmp(word,"if") == 0)
		    token = If;
		else if(strcmp(word,"then") == 0)
		    token = Then;
		else if(strcmp(word,"else") == 0)
		    token = Else;
		else if(strcmp(word,"endif") == 0)
		    token = Endif;
		else if(strcmp(word,"while") == 0)
		    token = While;
		else if(strcmp(word,"foreach") == 0)
		    token = Foreach;
		else if(strcmp(word,"end") == 0)
		    token = End;
		else if(strcmp(word,"switch") == 0)
		    token = Switch;
		else if(strcmp(word,"case") == 0)
		    token = Case;
		else if(strcmp(word,"default") == 0)
		    token = Default;
		else if(strcmp(word,"breaksw") == 0)
		    token = Breaksw;
		else if(strcmp(word,"endsw") == 0)
		    token = Endsw;
		else if(strcmp(word,"break") == 0)
		    token = Break;
		else if(strcmp(word,"continue") == 0)
		    token = Continue;
		else if(strcmp(word,"goto") == 0)
		    token = Goto;
		else if(ch == ':'  &&  newline == On) {
		    append_ch();
		    get_ch();
		    token = Label;
		}

		newline = Off;
		return;
	    }

	    token = Str;
	    set_ch();
	    get_ch();
	    newline = Off;
	    return;
    }
}

void set_ch(void)
/* set the character just read */
{
    *word = ch;
    *(word + 1) = '\0';
    word_ptr = 1;
}

void append_ch(void)
/* append the character just read */
{
    *(word + word_ptr) = ch;
    *(word + word_ptr + 1) = '\0';
    word_ptr++;
}

void get_ch(void)
/* get a character */
{
    if(ch == 0){
	ch = p_getchar(input_stream);
	pch = p_getchar(input_stream);
    }else{
	ch = pch;
	pch = p_getchar(input_stream);
    }

    if(comm_flag == Off  &&  
       ch == '\\'  &&  pch == '\n'){	/* substitute '\' at the end of line to space */
	line_count++;
	ch = ' ';
	pch = p_getchar(input_stream);
	while(pch == ' '  ||  pch == '\t')
	    pch = p_getchar(input_stream);
    }

    if(ch == '\n')
	line_count++;
}

int p_getchar(FILE *fp)
/* read a character */
{
    int c = 0;
    while(((0 <= c  &&  c < ' '  &&  c != '\t'  &&  c != '\n')  ||  c == 0x7F)  &&  c != EOF){
	c = getc(fp);
    }
    return c;
}
