/*                                                                  */
/*  parser.c -- Translate C shell program to PADEL and print result */
/*                                                                  */
/*  csh2pad  by S. Yoshida                                          */
/*                                                                  */

#include "config.h"

#include <string.h>
#ifdef HAVE_MALLOC_H
# include <malloc.h>
#endif
#include <ctype.h>
#include "csh2pad.h"
#include "lib.h"

#define MAKE_STRING_WHILE(X)	{\
				    count_paren = 0;\
				    cat_flag_tmp = 0;\
				    key_tmp[0] = '\0';\
				    if(token == LL)\
					cat_flag_tmp = 1;\
				    set_str();\
				    get_token();\
				    while(X){\
					if(token == Ret){\
					    append(" ");\
					    *sp = '\0';\
					    do{\
						get_token();\
					    }while(token == Sp);\
					}else{\
					    if(token == Sp){\
						strcat(sp,word);\
						if(cat_flag_tmp == 2)\
						    cat_flag_tmp = 0;\
					    }else{\
						if(token == Lpar)\
						     count_paren--;\
						if(token == Rpar)\
						     count_paren++;\
						if(cat_flag_tmp == 1)\
						    cat_flag_tmp = 2;\
						if(cat_flag_tmp == 2)\
						    strcat(key_tmp,word);\
						if(token == LL)\
						    cat_flag_tmp = 1;\
						if(*sp != '\0')\
						    append(sp);\
						*sp = '\0';\
						append_str();\
					    }\
					    get_token();\
					}\
				    }\
				    if(*sp != '\0')\
					append(sp);\
				    *sp = '\0';\
				}

#define END_OF_LINE()		{\
				    tt = token; get_token();\
				    if(strlen(key)==0){\
					while(token == Sp)\
					    get_token();\
					if(tt != Ret  &&  token == Ret)\
					    get_token();\
				    }\
				}

static char *str = NULL;
static int nest_level = 0;
static int count_paren = 0;
static token_type tt = Nontoken;
static char key_tmp[BUFSIZ] = {'\0'};
static int cat_flag_tmp = 0;

static void body(void);
static void command(void);
static void for_statement(void);
static void if_statement(void);
static void condition(void);

static void set_str(void);
static void append_str(void);
static void append(char *);
static void print_str(void);
static void print_hsp(void);

void parser(void)
/* output */
{
#ifdef DEBUG
    if(quiet == Off)
	fprintf(stderr,"Translating C shell program into PADEL starts...");
#endif

    puts("{");
    get_token();
    body();
    puts("}");

#ifdef DEBUG
    if(quiet == Off)
	fprintf(stderr,"done\n");
#endif
}

void body(void)
{
    char sp[BUFSIZ] = {'\0'};

    while(token != Eof){
	if (cat_flag > 0) {	/* such as "cat << EOF" */
	    puts("{");
	    while (cat_flag > 0  &&  token != Eof) {
		MAKE_STRING_WHILE(token != Ret  &&  token != Eof);
		if (strcmp(str,key) == 0) {
		    key[0] = '\0';
		    cat_flag = 0;
		    free(str);
		    str = NULL;
		} else {
		    printf("> ");
		    print_str();
		    putchar('\n');
		}
		get_token();
	    }
	    puts("}");
	}

	while(token == Sp)
	    get_token();

	if (token == End  ||  token == Else  ||  token == Endif  ||  token == Endsw  ||  token == Rpar)
	    return;

	command();
    }
}

void command(void)
{
    char sp[BUFSIZ] = {'\0'};
    char buf[BUFSIZ];
    int i,p;

    switch(token){
	case Foreach:
	case While:
	case Switch:
	    for_statement();
	    break;
	case If:
	    if_statement();
	    break;
	case Case:
	case Default:
	    MAKE_STRING_WHILE(token != Colon  &&  token != Eof);
	    nest_level--;
	    print_hsp();
	    nest_level++;
	    print_str();
	    printf(":\n");
	    get_token();
	    while(token == Sp  ||  token == Ret)
		get_token();
	    break;
	case Ret:
	    putchar('\n');
	    get_token();
	    break;
	case Semicolon:
	    print_hsp();
	    printf("{    }\n");
	    END_OF_LINE();
	    break;
	case Break:
	case Breaksw:
	case Continue:
	case Goto:
	    MAKE_STRING_WHILE(token != Semicolon  &&  token != Ret  &&  token != Eof);
	    print_hsp();
	    print_str();
	    putchar('\n');
	    END_OF_LINE();
	    break;
	case Lpar:
            putchar('\n');
            END_OF_LINE();
            nest_level++;
            body();             /* repetition body */
            nest_level--;
            END_OF_LINE();
            break;
	case Label:
	    set_str();
	    print_str();
	    putchar('\n');
	    END_OF_LINE();
	    break;
	default:
	    MAKE_STRING_WHILE(((token != Semicolon  &&  token != Ret  &&  token != Rpar  &&  token !=  And2  &&  token != Or2)  ||  count_paren < 0)  &&  token != Eof);
	    /* such as "cat << EOF" */
	    if (strlen(key_tmp) > 0) {
		if (key_tmp[0] == '"'  ||  key_tmp[0] == '\'') {
		    key_tmp[strlen(key_tmp) - 1] = '\0';
		    strcpy(key,key_tmp+1);
		} else {
		    strcpy(key,key_tmp);
		}
	    }
	    if (token == Ret)
		cat_flag = strlen(key);

	    print_hsp();
	    if(token == Rpar){
		print_str();
		putchar('\n');
		return;
	    }else if(token == And2  ||  token == Or2){  /* condition */
		sprintf(buf,"%s )",word);

		get_token();
		while (token == Sp  ||  token == Ret) {
		    get_token();
		}

		printf("if ( ");
		print_str();
		fputs(buf,stdout);

		if (token == Lpar) {
		    puts(" {");
		    END_OF_LINE();
		    nest_level++;
		    body();             /* repetition body*/
		    nest_level--;
		    print_hsp();
		    puts("}");
		} else {
		    putchar('\n');
		    nest_level++;
		    command();          /* repetition body */
		    nest_level--;
		    return;
		}
	    }else{
		print_str();
		putchar('\n');
		for(i = 0 ; i < count_paren ; i++)
		    putchar('\n');
	    }
	    END_OF_LINE();
    }
}

void for_statement(void)
{
    int i;
    char sp[BUFSIZ] = {'\0'};
    token_type type;

    type = token;

    set_str();
    print_hsp();

    if (type == Foreach)
	printf("for");
    else
	print_str();

    get_token();
    while (token == Sp) {
	strcat(sp,word);
	get_token();
    }
    fputs(sp,stdout);

    if (type == Foreach) {
	putchar('(');
	MAKE_STRING_WHILE(token != Ret  &&  token != Semicolon  &&  token != Eof);
	i = strlen(str) - 1;
	while(isspace(str[i])  &&  i >= 0)
	    str[i--] = '\0';
	print_str();
	printf(")%s{\n",sp);
    } else {
	condition();
	print_str();
	printf("%s{\n",sp);
	while (token != Ret  &&  token != Semicolon  &&  token != Eof)
	    get_token();
    }
    END_OF_LINE();

    nest_level++;
    if(type == Switch)
	nest_level++;
    body();				/* repetition body */
    nest_level--;
    if(type == Switch)
	nest_level--;

    print_hsp();
    puts("}");

    while (token != Ret  &&  token != Semicolon  &&  token != Eof)
	get_token();
    END_OF_LINE();
}

void if_statement(void)
{
    char sp[BUFSIZ] = {'\0'};

    set_str();

    if(elseif_flag == On)	/*spacing at the head of line is not required with "else if" */
	elseif_flag = Off;
    else
	print_hsp();

    print_str();

    get_token();
    while (token == Sp) {
	strcat(sp,word);
	get_token();
    }
    fputs(sp,stdout);

    condition();
    print_str();

    sp[0] = '\0';
    while (token == Sp) {
	strcat(sp,word);
	get_token();
    }

    if (token != Then) {
	putchar('\n');
	nest_level++;
	command();
	nest_level--;

	while (token == Sp)
	    get_token();
	if (token != Else)
	    return;
	else {
	    print_hsp();
	    printf("else");
	}
    } else {
	printf("%s{\n",sp);
	while (token != Ret  &&  token != Semicolon  &&  token != Eof)
	    get_token();
	END_OF_LINE();

	nest_level++;
	body();
	nest_level--;

	print_hsp();
	if (token == Endif) {
	    puts("}");
	    while (token != Ret  &&  token != Semicolon  &&  token != Eof)
		get_token();
	    END_OF_LINE();
	    return;
 	} else
	    printf("}%selse",sp);
    }

    sp[0] = '\0';
    no_print_comment_flag = On;
    get_token();
    while (token == Sp) {
	strcat(sp,word);
	get_token();
    }
    no_print_comment_flag = Off;

    if (token == If) {
	fputs(sp,stdout);
	elseif_flag = On;
	if_statement();
    } else {
	if (token == Comment)	/* comment right after "else" */
	    printf("%s\n%s\n{\n",sp,word);
	else
	    printf("%s{\n",sp);
	END_OF_LINE();

	nest_level++;
	body();
	nest_level--;

	print_hsp();
	puts("}");

	while (token != Ret  &&  token != Semicolon  &&  token != Eof)
	    get_token();
	END_OF_LINE();
    }
}

void condition(void)
{
    char sp[BUFSIZ] = {'\0'};

    append_str();
    get_token();

    while(token != Rpar  &&  token != Eof){
	if(token == Lpar)
	    condition();
	else{
	    if(token == Sp)
		strcat(sp,word);
	    else{
		if(*sp != '\0')
		    append(sp);
		*sp = '\0';
		append_str();
	    }

	    get_token();
	}
    }

    if(*sp != '\0')
	append(sp);
    append_str();
    get_token();
}

void set_str(void)
{
    int l;

    l = strlen(word);
    str = (char *)malloc((l+2) * sizeof(char));
    strcpy(str,word);
}

void append_str(void)
{
    int l;
    char *s;

    l = strlen(word);
    s = (char *)malloc((l+2) * sizeof(char));
    strcpy(s,word);
    str = joint(2,str,s);
}

void append(char *p)
{
    int l;
    char *s;

    l = strlen(p);
    s = (char *)malloc((l+2) * sizeof(char));
    strcpy(s,p);
    str = joint(2,str,s);
}

void print_str(void)
{
    if(str != NULL){
	printf("%s",str);
	free(str);
	str = NULL;
    }
}

void print_hsp(void)
{
    int i;

    for(i = 0 ; i < nest_level * tab ; i++)
	putchar(' ');
}
