/*                              */
/*  ifdef.c -- evaluate #ifdef  */
/*                              */
/*  c2pad  by S. Yoshida        */
/*                              */

#include "config.h"

#include <stdlib.h>
#include <string.h>
#ifdef HAVE_MALLOC_H
# include <malloc.h>
#endif
#include "c2pad.h"
#include "lib.h"

static int sp;
static int cp;
static int stack[50];
static token_type cmd[50];

static void remove_symbol(char *);
static DEF *is_defined(char *);
int is_operator(token_type);
static int rpn(void);
void evaluate_rpn(token_type);

void list_defines(void)
/* print list of defined symbols to be dealt with #ifdef */
{
    DEF *ptr;

    get_token();

    while(token != Eof){
	if(token == Sharp){		/* # at the beginning of line */
	    get_token();
	    while(token == Sp)
		get_token();

	    switch(token){
		case If:
		case Ifdef:
		case Ifndef:
		case Elif:
		    get_token();
		    while(token == Sp)
			get_token();

		    while(token != Ret){
			if(token == Name)
			    add_symbol(word,0);
			get_token();
			while(token == Sp)
			    get_token();
		    }
		    break;
		default:
		    while(token != Ret)
			get_token();
	    }
	}
	get_token();
    }

    for(ptr = defined_list ; ptr != NULL ; ptr = ptr->next)
	puts(ptr->sym);
}

void eval_ifdef(void)
/* evaluate #ifdef */
{
    token_type t;
    char tmp[BUFSIZ];
    int p;

    get_token();
    while(token == Sp)
	get_token();

    t = token;

    switch(t){
	case If:
	case Ifdef:
	case Ifndef:
	case Elif:
	case Define:
	case Undef:
	    get_token();
	    while(token == Sp)
		get_token();
	    break;
    }

    switch(t){
	case If:
	    p = rpn();
	    push(ifdef_value);
	    if(ifdef_value != TRUE)
		ifdef_value = ABS_FALSE;
	    else{
		if(p == 0)
		    ifdef_value = FALSE;
		else
		    ifdef_value = TRUE;
	    }
	    break;
	case Ifdef:
	    push(ifdef_value);
	    if(ifdef_value != TRUE)
		ifdef_value = ABS_FALSE;
	    else{
		if(is_defined(word) == NULL)
		    ifdef_value = FALSE;
		else
		    ifdef_value = TRUE;
	    }
	    break;
	case Ifndef:
	    push(ifdef_value);
	    if(ifdef_value != TRUE)
		ifdef_value = ABS_FALSE;
	    else{
		if(is_defined(word) == NULL)
		    ifdef_value = TRUE;
		else
		    ifdef_value = FALSE;
	    }
	    break;
	case Else:
	    if(ifdef_value == TRUE)
	        ifdef_value = FALSE;
	    else if(ifdef_value == FALSE)
	        ifdef_value = TRUE;
	    break;
	case Elif:
	    p = rpn();
	    if(ifdef_value == TRUE)
		ifdef_value = TRUED_FALSE;
	    else if(ifdef_value == FALSE){
		if(p == 0)
		    ifdef_value = FALSE;
		else
		    ifdef_value = TRUE;
	    }
	    break;
	case Endif:
	    ifdef_value = pop();
	    break;
	case Define:
	    strcpy(tmp,word);
	    get_token();
	    while(token == Sp)
		get_token();
	    if(token != Ret)
		p = rpn();
	    else
		p = 0;
	    if(ifdef_value == TRUE)
		add_symbol(tmp,p);
	    break;
	case Undef:
	    if(ifdef_value == TRUE)
		remove_symbol(word);
	    break;
    }

    while(token != Ret)
	get_token();

    if(ifdef_value == TRUE)
	get_token();
    else{
	while(ifdef_value != TRUE)
	    get_token();
    }
}

void add_symbol(char *sym,int val)
/* add newly defined symbol */
{
    DEF *ptr;
    int l;

    if(is_defined(sym) == NULL){
	ptr = defined_list;
	defined_list = (DEF *)malloc(sizeof(DEF));
	defined_list->next = ptr;

	l = strlen(sym);
	defined_list->sym = (char *)malloc((l+2) * sizeof(char));
	strcpy(defined_list->sym,sym);
	defined_list->val = val;
    }
}

void remove_symbol(char *sym)
/* delete defined symbol */
{
    DEF *prev = NULL;
    DEF *ptr;

    for(ptr = defined_list ; ptr != NULL ; ptr = ptr->next){
	if(strcmp(ptr->sym,sym) == 0){
	    if(prev != NULL)
	        prev->next = ptr->next;
	    else
	        defined_list = ptr->next;
	    free(ptr);
	    return;
	}
    }
}

DEF *is_defined(char *sym)
/* return pointer to requested symbol if already defined */
{
    DEF *ptr;

    for(ptr = defined_list ; ptr != NULL ; ptr = ptr->next){
	if(strcmp(ptr->sym,sym) == 0)
	    return(ptr);
    }
    return(NULL);
}

int is_operator(token_type t)
/* return weight if the token is operator */
{
    int p = -1;

    switch(token){
	case Defined:
	case Not:
	    p++;
	case Exp:
	    p++;
	case Times:
	case Divide:
	    p++;
	case Plus:
	case Minus:
	    p++;
	case Lshift:
	case Rshift:
	    p++;
	case Less:
	case Great:
	case Less_eq:
	case Great_eq:
	    p++;
	case Equal2:
	case Neq:
	    p++;
	case And:
	    p++;
	case Or:
	    p++;
	case And2:
	    p++;
	case Or2:
	    p++;
	case Rpar:
	    p++;
	case Lpar:
	    p++;
    }

    return(p);
}

int rpn(void)
/* calculate #ifdef condition in reversed Polland format */
{
    int p;
    DEF *ptr;

    sp = -1;
    cp = -1;

    while(token != Ret){
	if(token == Rpar){
	    while(cp > 0  &&  cmd[cp] != Lpar)
		evaluate_rpn(cmd[cp--]);
	    cp--;
	}else if(token == Lpar  ||  token == Not  ||  token == Defined){
	    /* always push left bracket and monomial operator */
	    cmd[++cp] = token;
	}else{
	    p = is_operator(token);
	    if(p != -1){			/* operator */
		while(cp != -1  &&  is_operator(cmd[cp]) >= p)
		    /* evaluate if the operator in stack is stronger or as same as the current opeator */
		    evaluate_rpn(cmd[cp--]);
		cmd[++cp] = token;		/* push into stack */
	    }else{
		if(token == Num){		/* number */
		    sp++;
		    stack[sp] = atoi(word);
		}else{				/* symbol */
		    sp++;
		    ptr = is_defined(word);
		    if(ptr != NULL)
			stack[sp] = ptr->val;
		    else
			stack[sp] = EOF;
		}
	    }
	}

	get_token();
	while(token == Sp)
	    get_token();
    }

    while(cp >= 0)
       evaluate_rpn(cmd[cp--]);

    return(stack[sp]);
}

void evaluate_rpn(token_type op)
/* evaluate operators in reversed Polland format */
{
    switch(op){
	case Equal2:
	    stack[sp-1] = stack[sp-1] == stack[sp];
	    sp--;
	    break;
	case Neq:
	    stack[sp-1] = stack[sp-1] != stack[sp];
	    sp--;
	    break;
	case Less:
	    stack[sp-1] = stack[sp-1] < stack[sp];
	    sp--;
	    break;
	case Great:
	    stack[sp-1] = stack[sp-1] > stack[sp];
	    sp--;
	    break;
	case Less_eq:
	    stack[sp-1] = stack[sp-1] <= stack[sp];
	    sp--;
	    break;
	case Great_eq:
	    stack[sp-1] = stack[sp-1] >= stack[sp];
	    sp--;
	    break;
	case Not:
	    if(stack[sp] == 0)
	        stack[sp] = 1;
	    else
	        stack[sp] = 0;
	    break;
	case And:
	    stack[sp-1] = stack[sp-1] & stack[sp];
	    sp--;
	    break;
	case Or:
	    stack[sp-1] = stack[sp-1] | stack[sp];
	    sp--;
	    break;
	case And2:
	    stack[sp-1] = stack[sp-1] && stack[sp];
	    sp--;
	    break;
	case Or2:
	    stack[sp-1] = stack[sp-1] || stack[sp];
	    sp--;
	    break;
	case Plus:
	    stack[sp-1] = stack[sp-1] + stack[sp];
	    sp--;
	    break;
	case Minus:
	    stack[sp-1] = stack[sp-1] - stack[sp];
	    sp--;
	    break;
	case Times:
	    stack[sp-1] = stack[sp-1] * stack[sp];
	    sp--;
	    break;
	case Divide:
	    stack[sp-1] = stack[sp-1] / stack[sp];
	    sp--;
	    break;
	case Exp:
	    stack[sp-1] = stack[sp-1] ^ stack[sp];
	    sp--;
	    break;
	case Lshift:
	    stack[sp-1] = stack[sp-1] << stack[sp];
	    sp--;
	    break;
	case Rshift:
	    stack[sp-1] = stack[sp-1] >> stack[sp];
	    sp--;
	    break;
	case Defined:
	    if(stack[sp] == EOF)
	        stack[sp] = 0;
	    else
	        stack[sp] = 1;
	    break;
    }
}
