/*                                 */
/*  token.c -- get token           */
/*                                 */
/*  awk2pad  by S. Yoshida         */
/*                                 */

#include "config.h"

#include <string.h>
#ifdef HAVE_MALLOC_H
# include <malloc.h>
#endif
#include <ctype.h>
#include "awk2pad.h"
#include "lib.h"

static int ch = 0;
static int pch = 0;		/* reading next character in ahead */
static int word_ptr = 0;	/* pointer to the tail of the word */
static bool newline = On;	/* whether it is a head of a line */
static bool comm_flag = Off;	/* whether it is in a comment field */

static void set_ch(void);
static void append_ch(void);
static void get_ch(void);
static int p_getchar(FILE *);

void get_token(void)
/* get token */
{
    if(token == Eof)
	fatal_error("input file terminates unexpectedly.");

    if(ch == 0)
	get_ch();

    switch(ch){
	case EOF:
	    token = Eof;
	    *word = '\0';
	    return;
	case '\n':
	    newline = On;
	    token = Ret;
	    *word = ' ';
	    *(word+1) = '\0';
	    get_ch();
	    return;
	case ' ':
	case '\t':
	    token = Sp;
	    set_ch();
	    get_ch();
	    while(ch == ' '  ||  ch == '\t'){
		append_ch();
		get_ch();
	    }
	    return;
	case '#':
	    comm_flag = On;
	    set_ch();
	    get_ch();
	    while(ch != '\n'){
		append_ch();
		get_ch();
	    }

	    /* do not print at this place */
	    if (no_print_comment_flag == On) {
		char buf[BUFSIZ];

		if(word[strlen(word)-1] == '\\'  ||  newline == On){
		    set_ch();
		    get_ch();
		}
		comm_flag = Off;
		newline = On;

		strcpy(buf,"// ");
		strcat(buf,word);
		strcat(buf,"\n");
		strcpy(word,buf);
		token = Comment;
		return;
	    }

	    if (elseif_flag == On) {
	        putchar('\n');
	        elseif_flag = Off;
	    }
	    printf("// ");
	    puts(word);
	    comm_flag = Off;
	    set_ch();
	    get_ch();
	    if(word[strlen(word)-1] != '\\'  &&  newline != On){	/* insert return code if the comment is not at the head of the line */
		newline = On;
		token = Ret;
	    }else{
		newline = On;
		get_token();
	    }
	    return;
	case '"':
	    newline = Off;
	    set_ch();

	    do{
		get_ch();
		if(ch == '\n'){
		    ch = '\\'; append_ch();
		    ch = 'n'; append_ch();
		}else if(ch == '\\'){
		    append_ch();
		    get_ch(); append_ch();
		    ch = ' ';
		}else
		    append_ch();
	    }while(ch != '"');

	    get_ch();

	    token = Str;
	    return;
	case '\'':
	    newline = Off;
	    set_ch();

	    do{
		get_ch();
		if(ch == '\n'){
		    ch = '\\'; append_ch();
		    ch = 'n'; append_ch();
		}else if(ch == '\\'){
		    append_ch();
		    get_ch(); append_ch();
		    ch = ' ';
		}else
		    append_ch();
	    }while(ch != '\'');

	    get_ch();

	    token = Str;
	    return;
	case '`':
	    newline = Off;
	    set_ch();

	    do{
		get_ch();
		if(ch == '\n'){
		    ch = '\\'; append_ch();
		    ch = 'n'; append_ch();
		}else if(ch == '\\'){
		    append_ch();
		    get_ch(); append_ch();
		    ch = ' ';
		}else
		    append_ch();
	    }while(ch != '`');

	    get_ch();

	    token = Str;
	    return;
	case '\\':
	    newline = Off;
	    token = Str;
	    set_ch();
	    get_ch();
	    append_ch();
	    get_ch();
	    return;
	case '(':
	    newline = Off;
	    token = Lpar;
	    set_ch();
	    get_ch();
	    return;
	case ')':
	    newline = Off;
	    token = Rpar;
	    set_ch();
	    get_ch();
	    return;
	case '{':
	    newline = Off;
	    token = Lbrace;
	    set_ch();
	    get_ch();
	    return;
	case '}':
	    newline = Off;
	    token = Rbrace;
	    set_ch();
	    get_ch();
	    return;
	case ';':
	    newline = Off;
	    token = Semicolon;
	    set_ch();
	    get_ch();
	    return;
	default:
	    if(isdigit(ch)){
		token = Num;
		set_ch();
		get_ch();
		while(isdigit(ch)){
		    append_ch();
		    get_ch();
		}
		newline = Off;
		return;
	    }

	    if(isalpha(ch)  ||  ch == '_'){
		token = Name;
		set_ch();
		get_ch();
		while(isalnum(ch)  ||  (char)ch == '_'){
		    append_ch();
		    get_ch();
		}

		if(strcmp(word,"if") == 0)
		    token = If;
		else if(strcmp(word,"else") == 0)
		    token = Else;
		else if(strcmp(word,"while") == 0)
		    token = While;
		else if(strcmp(word,"do") == 0)
		    token = Do;
		else if(strcmp(word,"for") == 0)
		    token = For;

		newline = Off;
		return;
	    }

	    token = Str;
	    set_ch();
	    get_ch();
	    newline = Off;
	    return;
    }
}

void set_ch(void)
/* set the character just read */
{
    *word = ch;
    *(word + 1) = '\0';
    word_ptr = 1;
}

void append_ch(void)
/* append the character just read */
{
    *(word + word_ptr) = ch;
    *(word + word_ptr + 1) = '\0';
    word_ptr++;
}

void get_ch(void)
/* get one character */
{
    if(ch == 0){
	ch = p_getchar(input_stream);
	pch = p_getchar(input_stream);
    }else{
	ch = pch;
	pch = p_getchar(input_stream);
    }

    if(comm_flag == Off  &&  
       ch == '\\'  &&  pch == '\n'){	/* substitute '\' at the end of the line to space */
	line_count++;
	ch = ' ';
	pch = p_getchar(input_stream);
	while(pch == ' '  ||  pch == '\t')
	    pch = p_getchar(input_stream);
    }

    if(ch == '\n')
	line_count++;
}

int p_getchar(FILE *fp)
/* read one character */
{
    int c = 0;
    while(((0 <= c  &&  c < ' '  &&  c != '\t'  &&  c != '\n')  ||  c == 0x7F)  &&  c != EOF){
	c = getc(fp);
    }
    return c;
}
